//
// pia.v
//
// ver.080815
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

module pia(
	clk,
	rst,
	addr,
	en,
	wen,
	wdata,
	rdata,
	irqa,
	irqb,
	pa_i,
	pa_o,
	ddra,
	ca1,
	ca2_i,
	ca2_o,
	pb_i,
	pb_o,
	ddrb,
	cb1,
	cb2_i,
	cb2_o
);

input	clk;
input	rst;
input [1:0]	addr;
input	en;
input	wen;
input [7:0]	wdata;
output [7:0]	rdata;
output	irqa;
output	irqb;
input [7:0]	pa_i;
output [7:0]	pa_o;
output [7:0]	ddra;
input	ca1;
input	ca2_i;
output	ca2_o;
input [7:0]	pb_i;
output [7:0]	pb_o;
output [7:0]	ddrb;
input	cb1;
input	cb2_i;
output	cb2_o;

reg [7:0]	rdata;

reg [7:0]	pra;
reg [7:0]	ddra;
reg	irqa1;
reg	irqa2;
reg [2:0]	ca2_ctl;
reg	pra_sel;
reg [1:0]	ca1_ctl;
wire [7:0]	cra;
wire	irqa1_en;
wire	irqa2_en;

reg	ca1_d;
reg	ca2_i_d;
reg	ca2_r;
wire	ca1_falling;
wire	ca1_rising;
wire	ca2_falling;
wire	ca2_rising;
wire	irqa1_set;
wire	irqa2_set;

reg [7:0]	prb;
wire [7:0]	pb_m;
reg [7:0]	ddrb;
reg	irqb1;
reg	irqb2;
reg [2:0]	cb2_ctl;
reg	prb_sel;
reg [1:0]	cb1_ctl;
wire [7:0]	crb;
wire	irqb1_en;
wire	irqb2_en;

reg	cb1_d;
reg	cb2_i_d;
reg	cb2_r;
wire	cb1_falling;
wire	cb1_rising;
wire	cb2_falling;
wire	cb2_rising;
wire	irqb1_set;
wire	irqb2_set;

wire	addr0_en;
wire	addr1_en;
wire	addr2_en;
wire	addr3_en;
wire	pra_wen;
wire	pra_ren;
wire	ddra_wen;
wire	prb_wen;
wire	prb_ren;
wire	ddrb_wen;
wire	cra_wen;
wire	crb_wen;


assign	addr0_en = (addr == 2'b00) & en;
assign	addr1_en = (addr == 2'b01) & en;
assign	addr2_en = (addr == 2'b10) & en;
assign	addr3_en = (addr == 2'b11) & en;

assign	pra_wen = addr0_en & pra_sel & wen;
assign	pra_ren = addr0_en & pra_sel & ~wen;
assign	ddra_wen = addr0_en & ~pra_sel & wen;
assign	prb_wen = addr2_en & prb_sel & wen;
assign	prb_ren = addr2_en & prb_sel & ~wen;
assign	ddrb_wen = addr2_en & ~prb_sel & wen;

assign	cra_wen = addr1_en & wen;
assign	crb_wen = addr3_en & wen;

// peripheral register A
always @(posedge clk or negedge rst) begin
	if(~rst)
		pra <= 8'h00;
	else if(pra_wen)
		pra <= wdata;
end

assign	pa_o = pra;

// data direction register A
always @(posedge clk or negedge rst) begin
	if(~rst)
		ddra <= 8'h00;
	else if(ddra_wen)
		ddra <= wdata;
end

// peripheral register B
always @(posedge clk or negedge rst) begin
	if(~rst)
		prb <= 8'h00;
	else if(prb_wen)
		prb <= wdata;
end

assign	pb_o = prb;

// data direction register B
always @(posedge clk or negedge rst) begin
	if(~rst)
		ddrb <= 8'h00;
	else if(ddrb_wen)
		ddrb <= wdata;
end

assign	pb_m = (ddrb & prb) | ((~ddrb) & pb_i);

// control register A
always @(posedge clk or negedge rst) begin
	if(~rst) begin
		ca2_ctl <= 3'b000;
		pra_sel <= 1'b0;
		ca1_ctl <= 2'b00;
	end
	else if(cra_wen) begin
		ca2_ctl <= wdata[5:3];
		pra_sel <= wdata[2];
		ca1_ctl <= wdata[1:0];
	end
end

assign	irqa1_en = ca1_ctl[0];
assign	irqa2_en = (~ca2_ctl[2]) & ca2_ctl[0];

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		ca1_d <= 1'b1;
		ca2_i_d <= 1'b1;
	end
	else begin
		ca1_d <= ca1;
		ca2_i_d <= ca2_i;
	end
end

assign	ca1_falling = ~ca1 & ca1_d;
assign	ca1_rising = ca1 & ~ca1_d;
assign	ca2_falling = ~ca2_i & ca2_i_d;
assign	ca2_rising = ca2_i & ~ca2_i_d;

assign	irqa1_set = (~ca1_ctl[1] & ca1_falling)	// falling edge
			| (ca1_ctl[1] & ca1_rising);	// rising edge

always @(posedge clk or negedge rst) begin
	if(~rst)
		irqa1 <= 1'b0;
	else if(irqa1_set)
		irqa1 <= 1'b1;
	else if(pra_ren)
		irqa1 <= 1'b0;
end

assign	irqa2_set = ((ca2_ctl[2:1] == 2'b00) & ca2_falling)	// falling edge
			| ((ca2_ctl[2:1] == 2'b01) & ca2_rising);	// rising edge

always @(posedge clk or negedge rst) begin
	if(~rst)
		irqa2 <= 1'b0;
	else if(irqa2_set)
		irqa2 <= 1'b1;
	else if(pra_ren)
		irqa2 <= 1'b0;
end

assign	cra = {irqa1, irqa2, ca2_ctl, pra_sel, ca1_ctl};

// CA2 output control
always @(posedge clk or negedge rst) begin
	if(~rst)
		ca2_r <= 1'b1;
	else if((ca2_ctl[2:1] == 2'b10) & pra_ren)
		ca2_r <= 1'b0;
	else if((ca2_ctl == 3'b100) & irqa1_set)
		ca2_r <= 1'b1;
	else if(ca2_ctl == 3'b101)
		ca2_r <= 1'b1;
	else if(ca2_ctl == 3'b110)
		ca2_r <= 1'b0;
	else if(ca2_ctl == 3'b111)
		ca2_r <= 1'b1;
end

assign	ca2_o = (ca2_ctl[2]) ? ca2_r : 1'b1;

// control register B
always @(posedge clk or negedge rst) begin
	if(~rst) begin
		cb2_ctl <= 3'b000;
		prb_sel <= 1'b0;
		cb1_ctl <= 2'b00;
	end
	else if(crb_wen) begin
		cb2_ctl <= wdata[5:3];
		prb_sel <= wdata[2];
		cb1_ctl <= wdata[1:0];
	end
end

assign	irqb1_en = cb1_ctl[0];
assign	irqb2_en = (~cb2_ctl[2]) & cb2_ctl[0];

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		cb1_d <= 1'b1;
		cb2_i_d <= 1'b1;
	end
	else begin
		cb1_d <= cb1;
		cb2_i_d <= cb2_i;
	end
end

assign	cb1_falling = ~cb1 & cb1_d;
assign	cb1_rising = cb1 & ~cb1_d;
assign	cb2_falling = ~cb2_i & cb2_i_d;
assign	cb2_rising = cb2_i & ~cb2_i_d;

assign	irqb1_set = (~cb1_ctl[1] & cb1_falling)	// falling edge
			| (cb1_ctl[1] & cb1_rising);	// rising edge

always @(posedge clk or negedge rst) begin
	if(~rst)
		irqb1 <= 1'b0;
	else if(irqb1_set)
		irqb1 <= 1'b1;
	else if(prb_ren)
		irqb1 <= 1'b0;
end

assign	irqb2_set = ((cb2_ctl[2:1] == 2'b00) & cb2_falling)	// falling edge
			| ((cb2_ctl[2:1] == 2'b01) & cb2_rising);	// rising edge

always @(posedge clk or negedge rst) begin
	if(~rst)
		irqb2 <= 1'b0;
	else if(irqb2_set)
		irqb2 <= 1'b1;
	else if(prb_ren)
		irqb2 <= 1'b0;
end

assign	crb = {irqb1, irqb2, cb2_ctl, prb_sel, cb1_ctl};

// CB2 output control
always @(posedge clk or negedge rst) begin
	if(~rst)
		cb2_r <= 1'b1;
	else if((cb2_ctl[2:1] == 2'b10) & prb_wen)
		cb2_r <= 1'b0;
	else if((cb2_ctl == 3'b100) & irqb1_set)
		cb2_r <= 1'b1;
	else if(cb2_ctl == 3'b101)
		cb2_r <= 1'b1;
	else if(cb2_ctl == 3'b110)
		cb2_r <= 1'b0;
	else if(cb2_ctl == 3'b111)
		cb2_r <= 1'b1;
end

assign	cb2_o = (cb2_ctl[2]) ? cb2_r : 1'b1;

// interrupt request
assign	irqa = (irqa1_en & irqa1) | (irqa2_en & irqa2);
assign	irqb = (irqb1_en & irqb1) | (irqb2_en & irqb2);

always @(
	addr or en
	or pa_i or ddra or pra_sel
	or pb_m or ddrb or prb_sel
	or cra or crb
) begin
	if(en) begin
		case(addr)
		2'b00:
			rdata = pra_sel ? pa_i : ddra;
		2'b01:
			rdata = cra;
		2'b10:
			rdata = prb_sel ? pb_m : ddrb;
		default:	// 2'b11
			rdata = crb;
		endcase
	end
	else begin
		rdata = 8'h00;
	end
end	// always comb

endmodule

// End of pia.v
