//
// ac68_fetch.v
//
// ac68 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac68_defs.v"

`define	FST_LEN		2
`define	FST_IDLE	2'b00
`define	FST_OPCODE	2'b01
`define	FST_OPRND	2'b10

module ac68_fetch(
	clk,
	rst,
	inst_rdy,
	inst_ack,
	pc_nxt,
	jmp_req,
	jmp_ack,
	new_pc,
	opcode,
	amode,
	oprnd,
	calc_ea,
	paddr,
	pen,
	prdata,
	pack
);

input	clk;
input	rst;
output	inst_rdy;
input	inst_ack;
output [15:0]	pc_nxt;
input	jmp_req;
output	jmp_ack;
input [15:0]	new_pc;
output [7:0]	opcode;
output [`AMODE_LEN - 1:0]	amode;
output [15:0]	oprnd;
output	calc_ea;
output [15:0]	paddr;
output	pen;
input [7:0]	prdata;
input	pack;

reg [`FST_LEN - 1:0]	state;
reg [`FST_LEN - 1:0]	state_nxt;

reg [15:0]	pc;
reg [15:0]	pc_nxt;
wire [15:0]	pc_plus_1;
wire [15:0]	pc_plus_2;

reg	inst_rdy;
reg	jmp_ack;

wire	opc_stat;
reg [7:0]	opcode_r;
reg [7:0]	opcode_nxt;

reg [`AMODE_LEN - 1:0]	amode_r;
reg [`AMODE_LEN - 1:0]	amode_nxt;

reg [15:0]	oprnd_r;
reg [15:0]	oprnd_nxt;

reg	calc_ea_r;
reg	calc_ea_nxt;

reg	code_pg;	// 0 - page2, 1 - page3
reg	code_pg_nxt;

reg	req_dat;
reg	req_dat_nxt;

wire [15:0]	paddr_i;
reg	pen_i;
reg	psize_i;
wire [15:0]	prdata_i;
wire	pack_i;


always @(posedge clk or negedge rst) begin
	if(~rst)
		state <= `FST_IDLE;
	else
		state <= state_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		pc <= 16'h0000;
	else
		pc <= pc_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		opcode_r <= `OPC_NOP;
		amode_r <= `AMODE_NONE;
		oprnd_r <= 16'h0000;
		calc_ea_r <= 1'b0;
	end
	else begin
		opcode_r <= opcode_nxt;
		amode_r <= amode_nxt;
		oprnd_r <= oprnd_nxt;
		calc_ea_r <= calc_ea_nxt;
	end
end

assign	opc_stat = (state == `FST_OPCODE);
assign	opcode = opc_stat ? opcode_nxt : opcode_r;
assign	amode = opc_stat ? amode_nxt : amode_r;
assign	oprnd = oprnd_nxt;
assign	calc_ea = opc_stat ? calc_ea_nxt : calc_ea_r;

always @(posedge clk or negedge rst) begin
	if(~rst)
		code_pg <= 1'b0;
	else
		code_pg <= code_pg_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		req_dat <= 1'b0;
	else
		req_dat <= req_dat_nxt;
end

assign	pc_plus_1 = pc + 16'h0001;
assign	pc_plus_2 = pc + 16'h0002;

always @(
	state or pc or pc_plus_1 or pc_plus_2 or new_pc
	or code_pg or opcode_r
	or amode_r
	or oprnd_r or calc_ea_r
	or req_dat or req_dat_nxt or prdata_i
	or inst_ack or jmp_req or pack_i
) begin
	pc_nxt = pc;
	pen_i = 1'b0;
	psize_i = 1'b0;	// 8bit
	req_dat_nxt = req_dat;
	code_pg_nxt = code_pg;
	opcode_nxt = opcode_r;
	amode_nxt = amode_r;
	oprnd_nxt = oprnd_r;
	calc_ea_nxt = calc_ea_r;
	inst_rdy = 1'b0;
	state_nxt = state;

	if(state == `FST_IDLE) begin
		if(jmp_req) begin
			pc_nxt = new_pc;
			jmp_ack = 1'b1;
			pen_i = 1'b1;
			state_nxt = `FST_OPCODE;
		end
		else begin
			jmp_ack = 1'b0;
			inst_rdy = 1'b1;
			if(inst_ack) begin
				pen_i = 1'b1;
				state_nxt = `FST_OPCODE;
			end
		end	// jmp_req - else
	end	// FST_IDLE
	else begin
		pen_i = 1'b0;

		if(pack_i) begin
			if(jmp_req) begin
				pc_nxt = new_pc;
				jmp_ack = 1'b1;
				pen_i = 1'b1;
				state_nxt = `FST_OPCODE;
			end
			else begin
				jmp_ack = 1'b0;
				if(state == `FST_OPCODE) begin
					opcode_nxt = prdata_i[7:0];
					amode_nxt = `AMODE_NONE;
					calc_ea_nxt = 1'b0;
					pc_nxt = pc_plus_1;

					if(
						(prdata_i[7:0] == 8'h8c)	// CPX 16bit immediate
						| (prdata_i[7:0] == 8'h8e)	// LDS 16bit immediate
						| (prdata_i[7:0] == 8'hce)	// LDX 16bit immediate
					) begin
						amode_nxt = `AMODE_IMM16;
					end
					else if(
						(prdata_i[7:4] == 4'h7)		// extended
						| (prdata_i[7:4] == 4'hb)	// extended
						| (prdata_i[7:4] == 4'hf)	// extended
					) begin
						amode_nxt = `AMODE_EXT;
					end
					else if(
						(prdata_i[7:0] == 8'h8d)	// BSR relative
						| (prdata_i[7:4] == 4'h2)	// relative
					) begin
						amode_nxt = `AMODE_REL;
					end
					else if(
						(prdata_i[7:4] == 4'h6)		// index
						| (prdata_i[7:4] == 4'ha)	// index
						| (prdata_i[7:4] == 4'he)	// index
					) begin
						amode_nxt = `AMODE_IDX;
						calc_ea_nxt = 1'b1;
					end
					else if(
						(prdata_i[7:4] == 4'h8)		// 8bit immediate
						| (prdata_i[7:4] == 4'hc)	// 8bit immediate
					) begin
						amode_nxt = `AMODE_IMM8;
					end
					else if(
						(prdata_i[7:4] == 4'h9)		// direct
						| (prdata_i[7:4] == 4'hd)	// direct
					) begin
						amode_nxt = `AMODE_DIR;
					end

					case(amode_nxt)
					`AMODE_IMM16,
					`AMODE_EXT: begin	// 16bit operand
						pen_i = 1'b1;
						psize_i = 1'b1;	// 16bit
						req_dat_nxt = 1'b1;
						state_nxt = `FST_OPRND;
					end
					`AMODE_REL,
					`AMODE_IDX,
					`AMODE_IMM8,
					`AMODE_DIR: begin	// 8bit operand
						pen_i = 1'b1;
						req_dat_nxt = 1'b0;
						state_nxt = `FST_OPRND;
					end
					default: begin
						inst_rdy = 1'b1;
						if(inst_ack) begin
							pen_i = 1'b1;
							state_nxt = `FST_OPCODE;
						end
						else begin
							state_nxt = `FST_IDLE;
						end
					end
					endcase
				end	// FST_OPCODE
				else if(state == `FST_OPRND) begin
					req_dat_nxt = 1'b0;
					oprnd_nxt = prdata_i;
					if(req_dat == 1'b1)
						pc_nxt = pc_plus_2;
					else
						pc_nxt = pc_plus_1;

					inst_rdy = 1'b1;
					if(inst_ack) begin
						pen_i = 1'b1;
						state_nxt = `FST_OPCODE;
					end
					else begin
						state_nxt = `FST_IDLE;
					end
				end	// FST_OPRND
			end	// jmp_req - else
		end
		else begin
			jmp_ack = 1'b0;
		end	// pack_i - else
	end	// FST_IDLE - else
end	// always comb

assign	paddr_i = pc_nxt;

ac68_mconr mconr0(
	.clk(clk),
	.rst(rst),
	.addr(paddr_i),
	.en(pen_i),
	.size(psize_i),
	.rdata(prdata_i),
	.ack(pack_i),
	.addr8(paddr),
	.en8(pen),
	.rdata8(prdata),
	.ack8(pack)
);

endmodule

// End of ac68_fetch.v
