//
// ac68_exe_data.v
//
// ac68 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac68_defs.v"

module ac68_exe_data(
	clk,
	rst,
	inst_ack,
	pc,
	new_pc,
	opcode,
	opcode_r,
	main_opr,
	cond_ok,
	rlst_end,
	amode,
	amode_r,
	oprnd,
	ea_mux,
	pc_mux,
	cpc_mux,
	tmp_mux,
	ina_mux,
	inb_mux,
	cc_mux,
	h_bit,
	c_bit,
	v_bit,
	new_h,
	new_n,
	new_z,
	new_v,
	new_c,
	opsize,
	ina,
	inb,
	outa,
	outb,
	rbus_mux,
	rnum_mux,
	rwd_mux,
	dbus_mux,
	da_mux,
	dwd_mux,
	daddr,
	den,
	dsize,
	dwen,
	dwdata,
	drdata,
	irq,
	nmi,
	irq_i,
	nmi_i
);

input	clk;
input	rst;
input	inst_ack;
input [15:0]	pc;
output [15:0]	new_pc;
input [7:0]	opcode;
output [7:0]	opcode_r;
output [`ALUOP_LEN - 1:0]	main_opr;
output	cond_ok;
output	rlst_end;
input [`AMODE_LEN - 1:0]	amode;
output [`AMODE_LEN - 1:0]	amode_r;
input [15:0]	oprnd;
input [`EAMUX_LEN - 1:0]	ea_mux;
input [`PCMUX_LEN - 1:0]	pc_mux;
input [`CPCMUX_LEN - 1:0]	cpc_mux;
input [`TMUX_LEN - 1:0]	tmp_mux;
input [`AMUX_LEN - 1:0]	ina_mux;
input [`BMUX_LEN - 1:0]	inb_mux;
input [`CCMUX_LEN - 1:0]	cc_mux;
output	h_bit;
output	c_bit;
output	v_bit;
input	new_h;
input	new_n;
input	new_z;
input	new_v;
input	new_c;
output	opsize;
output [15:0]	ina;
output [15:0]	inb;
input [15:0]	outa;
input [15:0]	outb;
input [`RMUX_LEN - 1:0]	rbus_mux;
input [`RNUM_LEN - 1:0]	rnum_mux;
input [`RWDMUX_LEN - 1:0]	rwd_mux;
input [`DMUX_LEN - 1:0]	dbus_mux;
input [`DAMUX_LEN - 1:0]	da_mux;
input [`DWDMUX_LEN - 1:0]	dwd_mux;
output [15:0]	daddr;
output	den;
output	dsize;
output	dwen;
output [15:0]	dwdata;
input [15:0]	drdata;
input	irq;
input	nmi;
output	irq_i;
output	nmi_i;

reg [15:0]	pc_r;
reg [7:0]	opcode_r;
reg	cond_ok;
reg [`AMODE_LEN - 1:0]	amode_r;
reg [15:0]	oprnd_r;
reg [`ALUOP_LEN - 1:0]	main_opr;
reg [`REG_LEN - 1:0]	opreg;
reg [`REG_LEN - 1:0]	opreg2;

reg [15:0]	cur_pc;
reg [15:0]	cur_pc_nxt;

reg [15:0]	effaddr;
reg [15:0]	effaddr_nxt;
reg [15:0]	effaddr_nxt2;

reg [15:0]	ina;
reg [15:0]	inb;

reg	h_bit;
reg	n_bit;
reg	z_bit;
reg	v_bit;
reg	c_bit;
reg	e_bit;
reg	f_bit;
reg	i_bit;
reg	h_nxt;
reg	n_nxt;
reg	z_nxt;
reg	v_nxt;
reg	c_nxt;
reg	e_nxt;
reg	f_nxt;
reg	i_nxt;
wire [7:0]	ccr;
wire [7:0]	ccr_nxt;

wire	irq_i;
wire	nmi_i;

wire [3:0]	cond;

wire [15:0]	ibase;

reg [`REG_LEN - 1:0]	raddr;
reg	ren;
reg	rwen;
reg [15:0]	rwdata;
wire [15:0]	rrdata;

reg [15:0]	daddr_i;
reg	den_i;
reg	dsize_i;
reg	dwen_i;
reg [15:0]	dwdata_i;
wire [15:0]	drdata_i;
wire	ack_i;

reg [15:0]	new_pc;

reg [15:0]	tmp;
reg [15:0]	tmp_nxt;


always @(posedge clk or negedge rst) begin
	if(~rst) begin
		pc_r <= 16'h0000;
		opcode_r <= `OPC_NOP;
		amode_r <= `AMODE_NONE;
		oprnd_r <= 16'h0000;
	end
	else if(inst_ack) begin
		pc_r <= pc;
		opcode_r <= opcode;
		amode_r <= amode;
		oprnd_r <= oprnd;
	end
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		cur_pc <= 16'h0000;
	else
		cur_pc <= cur_pc_nxt;
end

always @(cur_pc or cpc_mux or pc_r or pc) begin
	case(cpc_mux)
	`CPCMUX_PCR:
		cur_pc_nxt = pc_r;
	`CPCMUX_PCN:
		cur_pc_nxt = pc;
	default:
		cur_pc_nxt = cur_pc;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		effaddr <= 16'h0000;
	else
		effaddr <= effaddr_nxt2;
end

always @(
	effaddr_nxt or inst_ack or amode or oprnd
) begin
	if(inst_ack) begin
		case(amode)
		`AMODE_DIR:
			effaddr_nxt2 = {8'h00, oprnd[7:0]};
		`AMODE_EXT:
			effaddr_nxt2 = oprnd;
		default:
			effaddr_nxt2 = effaddr_nxt;
		endcase
	end
	else begin
		effaddr_nxt2 = effaddr_nxt;
	end
end	// always comb

always @(
	ea_mux or effaddr or outa or drdata_i or ibase
	or rrdata
) begin
	case(ea_mux)
	`EAMUX_OUTA:
		effaddr_nxt = outa;
	`EAMUX_DDAT:
		effaddr_nxt = drdata_i;
	`EAMUX_IBASE:
		effaddr_nxt = ibase;
	`EAMUX_RDAT:
		effaddr_nxt = rrdata;
	default:
		effaddr_nxt = effaddr;
	endcase
end	// always comb

always @(
	pc_mux or tmp or outa or effaddr
) begin
	case(pc_mux)
	`PCMUX_TMP:
		new_pc = tmp;
	`PCMUX_OUTA:
		new_pc = outa;
	`PCMUX_EA:
		new_pc = effaddr[15:0];
	default:	// PCMUX_ZERO
		new_pc = 16'h0000;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tmp <= 16'h0000;
	else
		tmp <= tmp_nxt;
end

always @(
	tmp_mux or tmp or drdata_i or rrdata
) begin
	case(tmp_mux)
	`TMUX_DDAT:
		tmp_nxt = drdata_i;
	`TMUX_RDAT:
		tmp_nxt = rrdata;
	`TMUX_PSHINI:
		tmp_nxt = `REG_IX;
	`TMUX_PSHNXT:
		if(tmp[`REG_LEN-1:0] == `REG_IX) tmp_nxt = `REG_ACCA;
		else if(tmp[`REG_LEN-1:0] == `REG_ACCA) tmp_nxt = `REG_ACCB;
		else if(tmp[`REG_LEN-1:0] == `REG_ACCB) tmp_nxt = `REG_CCR;
		else tmp_nxt = 3'b111;
	`TMUX_PULINI:
		tmp_nxt = `REG_CCR;
	`TMUX_PULNXT:
		if(tmp[`REG_LEN-1:0] == `REG_CCR) tmp_nxt = `REG_ACCB;
		else if(tmp[`REG_LEN-1:0] == `REG_ACCB) tmp_nxt = `REG_ACCA;
		else if(tmp[`REG_LEN-1:0] == `REG_ACCA) tmp_nxt = `REG_IX;
		else tmp_nxt = 3'b111;
	default:	// TMUX_NULL
		tmp_nxt = tmp;
	endcase
end	// always comb

assign	rlst_end = (tmp == 3'b111);

always @(opcode_r) begin
	if(opcode_r[7]) begin
		case(opcode_r[3:0])
		4'h0:
			main_opr = `ALUOP_SUB;
		4'h2:
			main_opr = `ALUOP_SBC;
		4'h4:
			main_opr = `ALUOP_AND;
		4'h8:
			main_opr = `ALUOP_EOR;
		4'h9:
			main_opr = `ALUOP_ADC;
		4'ha:
			main_opr = `ALUOP_OR;
		4'hb:
			main_opr = `ALUOP_ADD;
		default:
			main_opr = `ALUOP_NONE;
		endcase
	end
	else if(opcode_r[6]) begin
		case(opcode_r[3:0])
		4'h0:
			main_opr = `ALUOP_NEG;
		4'h3:
			main_opr = `ALUOP_COM;
		4'h4:
			main_opr = `ALUOP_LSR;
		4'h6:
			main_opr = `ALUOP_ROR;
		4'h7:
			main_opr = `ALUOP_ASR;
		4'h8:
			main_opr = `ALUOP_ASL;
		4'h9:
			main_opr = `ALUOP_ROL;
		4'ha:
			main_opr = `ALUOP_DEC;
		4'hc:
			main_opr = `ALUOP_INC;
		default:
			main_opr = `ALUOP_NONE;
		endcase
	end
	else if(opcode_r[5:4] == 2'b00) begin
		if(opcode_r[0])			// DEX (8'h09)
			main_opr = `ALUOP_DEC;
		else					// INX (8'h08)
			main_opr = `ALUOP_INC;
	end
	else if(opcode_r[5:4] == 2'b01) begin
		case(opcode_r[3:0])
		4'h0:
			main_opr = `ALUOP_SUB;
		4'hb:
			main_opr = `ALUOP_ADD;
		default:
			main_opr = `ALUOP_NONE;
		endcase
	end
	else if(opcode_r[5:4] == 2'b11) begin
		if(opcode_r[0])			// INS (8'h31)
			main_opr = `ALUOP_INC;
		else					// DES (8'h34)
			main_opr = `ALUOP_DEC;
	end
	else begin
		main_opr = `ALUOP_NONE;
	end
end	// always comb

always @(opcode_r) begin
	case(opcode_r[7:4])
	4'h0: begin
		if(opcode_r[3:0] == 4'h6)
			opreg = `REG_CCR;
		else if(opcode_r[3:0] == 4'h7)
			opreg = `REG_ACCA;
		else if(opcode_r[3:1] == 3'b100)	// 4'h8,4'h9
			opreg = `REG_IX;
		else
			opreg = `REG_ACCA;	// dummy 8bit
	end
	4'h1: begin
		if(opcode_r[3:0] == 4'h6)	// TAB
			opreg = `REG_ACCB;
		else						// TBA,ABA,SBA,CBA,DAA
			opreg = `REG_ACCA;
	end
	4'h3: begin
		if(opcode_r[3:0] == 4'h0)		// TSX
			opreg = `REG_IX;
		else	// INS,DES,PSH,PUL,TXS,RTS,RTI,WAI,SWI
			opreg = `REG_SP;
	end
	4'h4:
		opreg = `REG_ACCA;
	4'h8,4'h9,4'ha,4'hb: begin
		if(opcode_r[3:0] == 4'hc)
			opreg = `REG_IX;
		else if(opcode_r[3:0] == 4'hd)
			opreg = `REG_SP;
		else if(opcode_r[3:1] == 3'b111)
			opreg = `REG_SP;
		else
			opreg = `REG_ACCA;
	end
	4'h5:
		opreg = `REG_ACCB;
	4'hc,4'hd,4'he,4'hf: begin
		if(opcode_r[3:1] == 3'b111)
			opreg = `REG_IX;
		else
			opreg = `REG_ACCB;
	end
	default: begin
		opreg = `REG_ACCA;	// dummy 8bit
	end
	endcase
end	// always comb

always @(opcode_r) begin
	case(opcode_r[7:4])
	4'h0: begin
		if(opcode_r[3:0] == 4'h6)
			opreg2 = `REG_ACCA;
		else if(opcode_r[3:0] == 4'h7)
			opreg2 = `REG_CCR;
		else
			opreg2 = `REG_ACCA;	// dummy 8bit
	end
	4'h1: begin
		if(opcode_r[3:0] == 4'h6)	// TAB
			opreg2 = `REG_ACCA;
		else						// TBA,ABA,SBA,CBA
			opreg2 = `REG_ACCB;
	end
	4'h3: begin
		if(opcode_r[3:0] == 4'h0)		// TSX
			opreg2 = `REG_SP;
		else if(opcode_r[3:0] == 4'h5)	// TXS
			opreg2 = `REG_IX;
		else if(opcode_r[0])			// PSH,PUL
			opreg2 = `REG_ACCB;
		else							// PSH,PUL
			opreg2 = `REG_ACCA;
	end
	default:
		opreg2 = `REG_ACCA;	// dummy 8bit
	endcase
end	// always comb

assign	opsize = ~(opreg[`REG_LEN-1]);

always @(
	ina_mux or oprnd_r or rrdata or pc_r or drdata_i
	or effaddr or ibase or tmp
) begin
	case(ina_mux)
	`AMUX_OPRND:
		ina = oprnd_r;
	`AMUX_RDAT:
		ina = rrdata;
	`AMUX_PC:
		ina = pc_r;
	`AMUX_DDAT:
		ina = drdata_i;
	`AMUX_EA:
		ina = effaddr;
	`AMUX_IBASE:
		ina = ibase;
	`AMUX_TMP:
		ina = tmp;
	default:
		ina = 16'h0000;
	endcase
end	// always comb

always @(
	inb_mux or oprnd_r or drdata_i
	or tmp or rrdata
) begin
	case(inb_mux)
	`BMUX_OPRND:
		inb = oprnd_r;
	`BMUX_OPRND8SE:
		inb = {{8{oprnd_r[7]}},oprnd_r[7:0]};
	`BMUX_DDAT:
		inb = drdata_i;
	`BMUX_ONE:
		inb = 16'h0001;
	`BMUX_TWO:
		inb = 16'h0002;
	`BMUX_TMP:
		inb = tmp;
	`BMUX_OPRND8ZE:
		inb = {8'h00,oprnd_r[7:0]};
	`BMUX_RDAT:
		inb = rrdata;
	`BMUX_PSH,
	`BMUX_PUL:
		inb = (tmp[`REG_LEN-1]) ? 16'h0001 : 16'h0002;
	default:
		inb = 16'h0000;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		h_bit <= 1'b0;
		n_bit <= 1'b0;
		z_bit <= 1'b0;
		v_bit <= 1'b0;
		c_bit <= 1'b0;
		e_bit <= 1'b0;
		f_bit <= 1'b1;
		i_bit <= 1'b1;
	end
	else begin
		h_bit <= h_nxt;
		n_bit <= n_nxt;
		z_bit <= z_nxt;
		v_bit <= v_nxt;
		c_bit <= c_nxt;
		e_bit <= e_nxt;
		f_bit <= f_nxt;
		i_bit <= i_nxt;
	end
end

assign	ccr = {e_bit,f_bit,h_bit,i_bit,n_bit,z_bit,v_bit,c_bit};

always @(
	cc_mux or ccr_nxt
	or new_h or new_n or new_z or new_v or new_c
) begin
	h_nxt = ccr_nxt[5];
	n_nxt = ccr_nxt[3];
	z_nxt = ccr_nxt[2];
	v_nxt = ccr_nxt[1];
	c_nxt = ccr_nxt[0];
	e_nxt = ccr_nxt[7];
	f_nxt = ccr_nxt[6];
	i_nxt = ccr_nxt[4];

	case(cc_mux)
	`CCMUX_LOP: begin
		n_nxt = new_n;
		z_nxt = new_z;
		v_nxt = new_v;
	end
	`CCMUX_AOP: begin
		h_nxt = new_h;
		n_nxt = new_n;
		z_nxt = new_z;
		v_nxt = new_v;
		c_nxt = new_c;
	end
	`CCMUX_AOP2: begin
		n_nxt = new_n;
		z_nxt = new_z;
		v_nxt = new_v;
		c_nxt = new_c;
	end
	`CCMUX_ZBIT: begin
		z_nxt = new_z;
	end
	`CCMUX_AOP3: begin
		z_nxt = new_z;
		c_nxt = new_c;
	end
	`CCMUX_CLC: begin
		c_nxt = 1'b0;
	end
	`CCMUX_CLI: begin
		i_nxt = 1'b0;
	end
	`CCMUX_CLV: begin
		v_nxt = 1'b0;
	end
	`CCMUX_SEC: begin
		c_nxt = 1'b1;
	end
	`CCMUX_SEI: begin
		i_nxt = 1'b1;
	end
	`CCMUX_SEV: begin
		v_nxt = 1'b1;
	end
	endcase
end

assign	cond = opcode_r[3:0];

always @(
	cond or n_bit or z_bit or v_bit or c_bit
) begin
	case(cond[3:1])
	3'b001:		// bls (c|z = 1)
		cond_ok = (c_bit | z_bit);
	3'b010:		// bcs
		cond_ok = c_bit;
	3'b011:		// beq
		cond_ok = z_bit;
	3'b100:		// bvs
		cond_ok = v_bit;
	3'b101:		// bmi
		cond_ok = n_bit;
	3'b110:		// blt (n^v = 1)
		cond_ok = (n_bit ^ v_bit);
	3'b111:		// ble (z|(n^v) = 1)
		cond_ok = (z_bit | (n_bit ^ v_bit));
	default:	// never
		cond_ok = 1'b0;
	endcase

	if(cond[0] == 1'b0)
		cond_ok = ~cond_ok;
end	// always comb

assign	irq_i = i_bit ? 1'b0 : irq;
assign	nmi_i = nmi;

always @(rbus_mux) begin
	ren = 1'b0;
	rwen = 1'b0;

	case(rbus_mux)
	`RMUX_WR: begin
		ren = 1'b1;
		rwen = 1'b1;
	end
	`RMUX_RD: begin
		ren = 1'b1;
	end
	endcase
end	// always comb

always @(
	rnum_mux or opreg or opreg2 or tmp
) begin
	raddr = {(`REG_LEN){1'b0}};

	case(rnum_mux)
	`RNUM_OPREG:
		raddr = opreg;
	`RNUM_OPREG2:
		raddr = opreg2;
	`RNUM_TMP:
		raddr = tmp[`REG_LEN-1:0];
	`RNUM_SP:
		raddr = `REG_SP;
	endcase
end	// always comb

always @(
	rwd_mux or outa or rrdata or tmp
	or effaddr or drdata_i
) begin
	case(rwd_mux)
	`RWDMUX_OUTA:
		rwdata = outa;
	`RWDMUX_RDAT:
		rwdata = rrdata;
	`RWDMUX_TMP:
		rwdata = tmp;
	`RWDMUX_EA:
		rwdata = effaddr;
	`RWDMUX_DDAT:
		rwdata = drdata_i;
	default:
		rwdata = 16'h0000;
	endcase
end	// always comb

ac68_exe_regfile exe_regf0(
	.clk(clk),
	.rst(rst),
	.raddr(raddr),
	.ren(ren),
	.rwen(rwen),
	.rwdata(rwdata),
	.rrdata(rrdata),
	.ccr_i(ccr),
	.ccr_o(ccr_nxt),
	.ibase(ibase)
);

always @(
	dbus_mux or opreg or tmp
) begin
	den_i = 1'b0;
	dsize_i = 1'b0;	// 8bit
	dwen_i = 1'b0;

	case(dbus_mux)
	`DMUX_RD16: begin
		den_i = 1'b1;
		dsize_i = 1'b1;
	end
	`DMUX_RD: begin
		den_i = 1'b1;
		if(opreg[`REG_LEN-1] == 1'b0)	// 16bit
			dsize_i = 1'b1;
	end
	`DMUX_WR: begin
		den_i = 1'b1;
		if(opreg[`REG_LEN-1] == 1'b0)	// 16bit
			dsize_i = 1'b1;
		dwen_i = 1'b1;
	end
	`DMUX_WR16: begin
		den_i = 1'b1;
		dsize_i = 1'b1;
		dwen_i = 1'b1;
	end
	`DMUX_WRT: begin
		den_i = 1'b1;
		dsize_i = ~(tmp[`REG_LEN-1]);
		dwen_i = 1'b1;
	end
	`DMUX_RDT: begin
		den_i = 1'b1;
		dsize_i = ~(tmp[`REG_LEN-1]);
	end
	`DMUX_RD8: begin
		den_i = 1'b1;
	end
	`DMUX_WR8: begin
		den_i = 1'b1;
		dwen_i = 1'b1;
	end
	endcase
end	// always comb

always @(
	da_mux or effaddr or outa or rrdata
) begin
	case(da_mux)
	`DAMUX_RSTV:
		daddr_i = 16'hfffe;
	`DAMUX_EA:
		daddr_i = effaddr;
	`DAMUX_OUTA:
		daddr_i = outa;
	`DAMUX_RDAT:
		daddr_i = rrdata;
	`DAMUX_NMIV:
		daddr_i = 16'hfffc;
	`DAMUX_IRQV:
		daddr_i = 16'hfff8;
	`DAMUX_SWIV:
		daddr_i = 16'hfffa;
	default:	// DAMUX_ZERO
		daddr_i = 16'h0000;
	endcase
end	// always comb

always @(
	dwd_mux or outa or pc_r or rrdata
	or cur_pc
) begin
	case(dwd_mux)
	`DWDMUX_OUTA:
		dwdata_i = outa;
	`DWDMUX_PC:
		dwdata_i = pc_r;
	`DWDMUX_RDAT:
		dwdata_i = rrdata;
	`DWDMUX_CPC:
		dwdata_i = cur_pc;
	default:
		dwdata_i = 16'h0000;
	endcase
end	// always comb

assign	daddr = daddr_i;
assign	den = den_i;
assign	dsize = dsize_i;
assign	dwen = dwen_i;
assign	dwdata = dwdata_i;
assign	drdata_i = drdata;

endmodule

// End of ac68_exe_data.v
