//
// ac68_exe_ctrl.v
//
// ac68 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac68_defs.v"

`define	EST_LEN		4
`define	EST_IDLE	4'b0000
`define	EST_AM		4'b0001
`define	EST_EXE1	4'b0010
`define	EST_EXE2	4'b0011
`define	EST_EXE3	4'b0100
`define	EST_EXE4	4'b0101
`define	EST_EXE5	4'b0110
`define	EST_EXE6	4'b0111
`define	EST_EXE7	4'b1000
`define	EST_EXE8	4'b1001

`define	EXST_LEN	3
`define	EXST_NONE	3'b000
`define	EXST_INIT	3'b001
`define	EXST_RST	3'b010
`define	EXST_IVEC	3'b011

module ac68_exe_ctrl(
	clk,
	rst,
	inst_rdy,
	inst_ack,
	jmp_req,
	jmp_ack,
	opcode,
	cond_ok,
	rlst_end,
	amode,
	calc_ea,
	ea_mux,
	pc_mux,
	cpc_mux,
	tmp_mux,
	op_sel,
	ina_mux,
	inb_mux,
	cc_mux,
	rbus_mux,
	rnum_mux,
	rwd_mux,
	dbus_mux,
	da_mux,
	dwd_mux,
	ack,
	irq,
	nmi
);

input	clk;
input	rst;
input	inst_rdy;
output	inst_ack;
output	jmp_req;
input	jmp_ack;
input [7:0]	opcode;
input	cond_ok;
input	rlst_end;
input [`AMODE_LEN - 1:0]	amode;
input	calc_ea;
output [`EAMUX_LEN - 1:0]	ea_mux;
output [`PCMUX_LEN - 1:0]	pc_mux;
output [`CPCMUX_LEN - 1:0]	cpc_mux;
output [`TMUX_LEN - 1:0]	tmp_mux;
output [`ALUOP_LEN -1:0]	op_sel;
output [`AMUX_LEN - 1:0]	ina_mux;
output [`BMUX_LEN - 1:0]	inb_mux;
output [`CCMUX_LEN - 1:0]	cc_mux;
output [`RMUX_LEN - 1:0]	rbus_mux;
output [`RNUM_LEN - 1:0]	rnum_mux;
output [`RWDMUX_LEN - 1:0]	rwd_mux;
output [`DMUX_LEN - 1:0]	dbus_mux;
output [`DAMUX_LEN - 1:0]	da_mux;
output [`DWDMUX_LEN - 1:0]	dwd_mux;
input	ack;
input	irq;
input	nmi;

reg	inst_req;

reg	jmp_req;
reg	jmp_fetch;
reg	jmp_fetch_nxt;

reg [`EAMUX_LEN - 1:0]	ea_mux;
reg [`PCMUX_LEN - 1:0]	pc_mux;
reg [`CPCMUX_LEN - 1:0]	cpc_mux;
reg [`TMUX_LEN - 1:0]	tmp_mux;
reg [`ALUOP_LEN - 1:0]	op_sel;
reg [`AMUX_LEN - 1:0]	ina_mux;
reg [`BMUX_LEN - 1:0]	inb_mux;
reg [`CCMUX_LEN - 1:0]	cc_mux;
reg [`RMUX_LEN - 1:0]	rbus_mux;
reg [`RNUM_LEN - 1:0]	rnum_mux;
reg [`RWDMUX_LEN - 1:0]	rwd_mux;
reg [`DMUX_LEN - 1:0]	dbus_mux;
reg [`DAMUX_LEN - 1:0]	da_mux;
reg [`DWDMUX_LEN - 1:0]	dwd_mux;

reg [`EST_LEN - 1:0]	state;
reg [`EST_LEN - 1:0]	state_nxt;
wire [`EST_LEN - 1:0]	state_def;

reg [`EXST_LEN - 1:0]	exstate;
reg [`EXST_LEN - 1:0]	exstate_nxt;

reg	int_wai;
wire	iacpt_timing;
wire	int_exe;
reg	irq_exe;
reg	nmi_d;
reg	nmi_r;
reg	nmi_exe;


always @(posedge clk or negedge rst) begin
	if(~rst)
		state <= `EST_EXE1;
	else
		state <= state_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		exstate <= `EXST_INIT;
	else
		exstate <= exstate_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		jmp_fetch <= 1'b0;
	else
		jmp_fetch <= jmp_fetch_nxt;
end

always @(jmp_fetch or inst_ack or jmp_req) begin
	if(inst_ack)
		jmp_fetch_nxt = 1'b0;
	else if(jmp_req)
		jmp_fetch_nxt = 1'b1;
	else
		jmp_fetch_nxt = jmp_fetch;
end	// always comb

always @(jmp_req or inst_req or jmp_fetch) begin
	if(jmp_req) cpc_mux = `CPCMUX_PCN;
	else begin
		if(jmp_fetch) cpc_mux = `CPCMUX_NULL;
		else begin
			if(inst_req) cpc_mux = `CPCMUX_PCR;
			else cpc_mux = `CPCMUX_NULL;
		end
	end
end	// always comb

assign	iacpt_timing = inst_req | int_wai;
assign	int_exe = (irq_exe | nmi_exe);

always @(posedge clk or negedge rst) begin
	if(~rst)
		irq_exe <= 1'b0;
	else if(~int_exe & ~nmi_r & irq & iacpt_timing)
		irq_exe <= 1'b1;
	else if(jmp_ack)
		irq_exe <= 1'b0;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		nmi_d <= 1'b0;
	else
		nmi_d <= nmi;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		nmi_r <= 1'b0;
	else if((~nmi_d) & nmi)
		nmi_r <= 1'b1;
	else if(nmi_exe)
		nmi_r <= 1'b0;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		nmi_exe <= 1'b0;
	else if(~int_exe & nmi_r & iacpt_timing)
		nmi_exe <= 1'b1;
	else if(jmp_ack)
		nmi_exe <= 1'b0;
end

assign	state_def = inst_ack
			? ((calc_ea) ? (`EST_AM) : (`EST_EXE1))
			: (`EST_IDLE);

assign	inst_ack = inst_req & inst_rdy;

always @(
	state or state_def or exstate
	or opcode or amode
	or cond_ok
	or inst_ack or ack or jmp_ack
	or irq_exe or nmi_exe or int_exe
	or int_wai or rlst_end
) begin
	inst_req = 1'b0;
	jmp_req = 1'b0;
	int_wai = 1'b0;
	ea_mux = `EAMUX_NULL;
	pc_mux = `PCMUX_ZERO;
	tmp_mux = `TMUX_NULL;
	op_sel = `ALUOP_NONE;
	ina_mux = `AMUX_ZERO;
	inb_mux = `BMUX_ZERO;
	cc_mux = `CCMUX_NULL;
	rbus_mux = `RMUX_NULL;
	rnum_mux = `RNUM_ZERO;
	rwd_mux = `RWDMUX_ZERO;
	dbus_mux = `DMUX_NULL;
	da_mux = `DAMUX_ZERO;
	dwd_mux = `DWDMUX_ZERO;
	state_nxt = state;
	exstate_nxt = exstate;

	if(exstate == `EXST_INIT) begin
		state_nxt = `EST_EXE1;
		exstate_nxt = `EXST_RST;
	end	// EXST_INIT
	else if(exstate == `EXST_RST) begin
		if(state == `EST_EXE1) begin
			dbus_mux = `DMUX_RD16;
			da_mux = `DAMUX_RSTV;
			state_nxt = `EST_EXE2;
		end
		else if(state == `EST_EXE2) begin
			if(ack) begin
				tmp_mux = `TMUX_DDAT;
				state_nxt = `EST_EXE3;
			end
		end
		else begin
			jmp_req = 1'b1;
			pc_mux = `PCMUX_TMP;
			if(jmp_ack) begin
				state_nxt = `EST_IDLE;
				exstate_nxt = `EXST_NONE;
			end
		end
	end	// EXST_RST
	else if(exstate == `EXST_IVEC) begin
		if(state == `EST_EXE1) begin
			rbus_mux = `RMUX_RD;
			rnum_mux = `RNUM_SP;
			tmp_mux = `TMUX_PSHINI;
			state_nxt = `EST_EXE2;
		end
		else if(state == `EST_EXE2) begin
			op_sel = `ALUOP_SUB16;
			ina_mux = `AMUX_RDAT;
			inb_mux = `BMUX_ONE;
			ea_mux = `EAMUX_OUTA;
			dbus_mux = `DMUX_WR16;
			da_mux = `DAMUX_OUTA;
			if(int_exe) dwd_mux = `DWDMUX_CPC;
			else dwd_mux = `DWDMUX_PC;
			state_nxt = `EST_EXE3;
		end
		else if(state == `EST_EXE3) begin
			if(ack) begin
				if(rlst_end) begin
					op_sel = `ALUOP_SUB16;
					ina_mux = `AMUX_EA;
					inb_mux = `BMUX_ONE;
					rbus_mux = `RMUX_WR;
					rnum_mux = `RNUM_SP;
					rwd_mux = `RWDMUX_OUTA;
					state_nxt = `EST_EXE5;
				end
				else begin
					op_sel = `ALUOP_SUB16;
					ina_mux = `AMUX_EA;
					inb_mux = `BMUX_PSH;
					ea_mux = `EAMUX_OUTA;
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_TMP;
					state_nxt = `EST_EXE4;
				end
			end
		end
		else if(state == `EST_EXE4) begin
			dbus_mux = `DMUX_WRT;
			da_mux = `DAMUX_EA;
			dwd_mux = `DWDMUX_RDAT;
			tmp_mux = `TMUX_PSHNXT;
			state_nxt = `EST_EXE3;
		end
		else if(state == `EST_EXE5) begin
			state_nxt = `EST_EXE6;
			dbus_mux = `DMUX_RD16;
			if(nmi_exe) begin
				da_mux = `DAMUX_NMIV;
				cc_mux = `CCMUX_SEI;
			end
			else if(irq_exe) begin
				da_mux = `DAMUX_IRQV;
				cc_mux = `CCMUX_SEI;
			end
			else if(opcode == `OPC_SWI) begin
				da_mux = `DAMUX_SWIV;
				cc_mux = `CCMUX_SEI;
			end
			else begin	// OPC_WAI
				dbus_mux = `DMUX_NULL;
				int_wai = 1'b1;
				state_nxt = `EST_EXE5;
			end
		end
		else if(state == `EST_EXE6) begin
			if(ack) begin
				ea_mux = `EAMUX_DDAT;
				state_nxt = `EST_EXE7;
			end
		end
		else begin
			jmp_req =1'b1;
			pc_mux = `PCMUX_EA;
			if(jmp_ack) begin
				state_nxt = `EST_IDLE;
				exstate_nxt = `EXST_NONE;
			end
		end
	end	// EXST_IVEC
	else if(int_exe) begin
		exstate_nxt = `EXST_IVEC;
		if(int_wai) state_nxt = `EST_EXE5;
		else state_nxt = `EST_EXE1;
	end
	else if(state == `EST_IDLE) begin
		inst_req = 1'b1;
		state_nxt = state_def;
	end	// EST_IDLE
	else begin
		// instructions without effective address calc
		if(opcode == `OPC_NOP) begin
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_NOP
		else if(opcode == `OPC_CLC) begin
			cc_mux = `CCMUX_CLC;
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_CLC
		else if(opcode == `OPC_CLI) begin
			cc_mux = `CCMUX_CLI;
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_CLI
		else if(opcode == `OPC_CLV) begin
			cc_mux = `CCMUX_CLV;
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_CLV
		else if(opcode == `OPC_SEC) begin
			cc_mux = `CCMUX_SEC;
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_SEC
		else if(opcode == `OPC_SEI) begin
			cc_mux = `CCMUX_SEI;
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_SEI
		else if(opcode == `OPC_SEV) begin
			cc_mux = `CCMUX_SEV;
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_SEV
		else if(opcode == `OPC_SWI) begin
			exstate_nxt = `EXST_IVEC;
			state_nxt = `EST_EXE1;
		end	// OPC_SWI
		else if(opcode == `OPC_WAI) begin
			exstate_nxt = `EXST_IVEC;
			state_nxt = `EST_EXE1;
		end	// OPC_WAI
		else if(opcode == `OPC_DAA) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_DAA;
				ina_mux = `AMUX_RDAT;
				cc_mux = `CCMUX_AOP2;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_DAA
		else if(opcode == `OPC_TSX) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG2;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_ADD16;
				ina_mux = `AMUX_RDAT;
				inb_mux = `BMUX_ONE;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_TSX
		else if(opcode == `OPC_TXS) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG2;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_SUB16;
				ina_mux = `AMUX_RDAT;
				inb_mux = `BMUX_ONE;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_TXS
		else if(opcode == `OPC_CBA) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				tmp_mux = `TMUX_RDAT;
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG2;
				state_nxt = `EST_EXE3;
			end
			else begin
				op_sel = `ALUOP_SUB;
				ina_mux = `AMUX_TMP;
				inb_mux = `BMUX_RDAT;
				cc_mux = `CCMUX_AOP;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_CBA
		else if(opcode == `OPC_TSTR) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_THRU;
				ina_mux = `AMUX_RDAT;
				cc_mux = `CCMUX_LOP;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_TSTR
		else if({opcode[7:5],opcode[3:0]} == `OPC_CLRR) begin
			op_sel = `ALUOP_THRU;
			ina_mux = `AMUX_ZERO;
			cc_mux = `CCMUX_AOP2;
			rbus_mux = `RMUX_WR;
			rnum_mux = `RNUM_OPREG;
			rwd_mux = `RWDMUX_OUTA;
			inst_req = 1'b1;
			state_nxt = state_def;
		end	// OPC_CLRR
		else if(opcode[7:1] == `OPC_TFR) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG2;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_THRU;
				ina_mux = `AMUX_RDAT;
				cc_mux = `CCMUX_LOP;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_TFR
		else if(opcode[7:1] == `OPC_TFR2) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG2;
				state_nxt = `EST_EXE2;
			end
			else begin
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_RDAT;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_TFR2
		else if(opcode == `OPC_RTS) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				tmp_mux = `TMUX_RDAT;
				op_sel = `ALUOP_ADD16;
				ina_mux = `AMUX_RDAT;
				inb_mux = `BMUX_ONE;
				dbus_mux = `DMUX_RD16;
				da_mux = `DAMUX_OUTA;
				state_nxt = `EST_EXE3;
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					op_sel = `ALUOP_ADD16;
					ina_mux = `AMUX_TMP;
					inb_mux = `BMUX_TWO;
					rbus_mux = `RMUX_WR;
					rnum_mux = `RNUM_OPREG;
					rwd_mux = `RWDMUX_OUTA;
					ea_mux = `EAMUX_DDAT;
					state_nxt = `EST_EXE4;
				end
			end
			else begin
				jmp_req = 1'b1;
				pc_mux = `PCMUX_EA;
				if(jmp_ack) begin
					state_nxt = `EST_IDLE;
				end
			end
		end	// OPC_RTS
		else if(opcode == `OPC_RTI) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				tmp_mux = `TMUX_PULINI;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				op_sel = `ALUOP_ADD16;
				ina_mux = `AMUX_RDAT;
				inb_mux = `BMUX_ONE;
				ea_mux = `EAMUX_OUTA;
				dbus_mux = `DMUX_RDT;
				da_mux = `DAMUX_OUTA;
				state_nxt = `EST_EXE3;
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					op_sel = `ALUOP_ADD16;
					ina_mux = `AMUX_EA;
					inb_mux = `BMUX_PUL;
					ea_mux = `EAMUX_OUTA;
					rbus_mux = `RMUX_WR;
					rnum_mux = `RNUM_TMP;
					rwd_mux = `RWDMUX_DDAT;
					tmp_mux = `TMUX_PULNXT;
					state_nxt = `EST_EXE4;
				end
			end
			else if(state == `EST_EXE4) begin
				if(rlst_end) begin
					op_sel = `ALUOP_ADD16;
					ina_mux = `AMUX_EA;
					inb_mux = `BMUX_ONE;
					dbus_mux = `DMUX_RD16;
					da_mux = `DAMUX_EA;
					rbus_mux = `RMUX_WR;
					rnum_mux = `RNUM_OPREG;
					rwd_mux = `RWDMUX_OUTA;
					state_nxt = `EST_EXE5;
				end
				else begin
					dbus_mux = `DMUX_RDT;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE3;
				end
			end
			else if(state_nxt == `EST_EXE5) begin
				if(ack) begin
					ea_mux = `EAMUX_DDAT;
					state_nxt = `EST_EXE6;
				end
			end
			else begin
				jmp_req = 1'b1;
				pc_mux = `PCMUX_EA;
				if(jmp_ack) begin
					state_nxt = `EST_IDLE;
					exstate_nxt = `EXST_NONE;
				end
			end
		end	// OPC_RTI
		else if(opcode[7:1] == `OPC_PUSH) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				ea_mux = `EAMUX_RDAT;
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG2;
				state_nxt = `EST_EXE3;
			end
			else if(state == `EST_EXE3) begin
				dbus_mux = `DMUX_WR8;
				da_mux = `DAMUX_EA;
				dwd_mux = `DWDMUX_RDAT;
				op_sel = `ALUOP_SUB16;
				ina_mux = `AMUX_EA;
				inb_mux = `BMUX_ONE;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				state_nxt = `EST_EXE4;
			end
			else begin
				if(ack) begin
					inst_req = 1'b1;
					state_nxt = state_def;
				end
			end
		end	// OPC_PUSH
		else if(opcode[7:1] == `OPC_PULL) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				op_sel = `ALUOP_ADD16;
				ina_mux = `AMUX_RDAT;
				inb_mux = `BMUX_ONE;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				dbus_mux = `DMUX_RD8;
				da_mux = `DAMUX_OUTA;
				state_nxt = `EST_EXE3;
			end
			else begin
				if(ack) begin
					rbus_mux = `RMUX_WR;
					rnum_mux = `RNUM_OPREG2;
					rwd_mux = `RWDMUX_DDAT;
					inst_req = 1'b1;
					state_nxt = state_def;
				end
			end
		end	// OPC_PULL
		else if(opcode[7:1] == `OPC_INDCX) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_MAIN;
				ina_mux = `AMUX_RDAT;
				cc_mux = `CCMUX_ZBIT;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_INDCX
		else if(opcode[7:3] == `OPC_INDCS) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_MAIN;
				ina_mux = `AMUX_RDAT;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_INDCS
		else if(opcode[7:4] == `OPC_BOPR) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				tmp_mux = `TMUX_RDAT;
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG2;
				state_nxt = `EST_EXE3;
			end
			else begin
				op_sel = `ALUOP_MAIN;
				ina_mux = `AMUX_TMP;
				inb_mux = `BMUX_RDAT;
				cc_mux = `CCMUX_AOP;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_BOPR
		else if({opcode[7:5],opcode[3:0]} == `OPC_TSTR) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_TST;
				ina_mux = `AMUX_RDAT;
				cc_mux = `CCMUX_AOP;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_TSTR
		else if(opcode[7:5] == `OPC_UOPR) begin
			if(state == `EST_EXE1) begin
				rbus_mux = `RMUX_RD;
				rnum_mux = `RNUM_OPREG;
				state_nxt = `EST_EXE2;
			end
			else begin
				op_sel = `ALUOP_MAIN;
				ina_mux = `AMUX_RDAT;
				cc_mux = `CCMUX_AOP;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// OPC_UOPR
		else if(
			(amode == `AMODE_IMM8) | (amode == `AMODE_IMM16)
		) begin
			if({opcode[7],opcode[3:0]} == `OPC_BIT) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					state_nxt = `EST_EXE2;
				end
				else begin
					op_sel = `ALUOP_AND;
					ina_mux = `AMUX_RDAT;
					inb_mux = `BMUX_OPRND;
					cc_mux = `CCMUX_LOP;
					inst_req = 1'b1;
					state_nxt = state_def;
				end
			end	// OPC_BIT
			else if({opcode[7],opcode[3:0]} == `OPC_CMP) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					state_nxt = `EST_EXE2;
				end
				else begin
					op_sel = `ALUOP_SUB;
					ina_mux = `AMUX_RDAT;
					inb_mux = `BMUX_OPRND;
					cc_mux = `CCMUX_AOP2;
					inst_req = 1'b1;
					state_nxt = state_def;
				end
			end	// OPC_CMP
			else if({opcode[7:6],opcode[3:0]} == `OPC_CMP16) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					state_nxt = `EST_EXE2;
				end
				else begin
					op_sel = `ALUOP_SUB;
					ina_mux = `AMUX_RDAT;
					inb_mux = `BMUX_OPRND;
					cc_mux = `CCMUX_LOP;
					inst_req = 1'b1;
					state_nxt = state_def;
				end
			end	// OPC_CMP16
			else if(
				({opcode[7],opcode[3:0]} == `OPC_LD)
				| ({opcode[7],opcode[3:0]} == `OPC_LD16)
			) begin
				op_sel = `ALUOP_THRU;
				ina_mux = `AMUX_OPRND;
				rbus_mux = `RMUX_WR;
				rnum_mux = `RNUM_OPREG;
				rwd_mux = `RWDMUX_OUTA;
				cc_mux = `CCMUX_LOP;
				inst_req = 1'b1;
				state_nxt = state_def;
			end	// OPC_LD,OPC_LD16
			else if(opcode[7] == `OPC_BOP) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					state_nxt = `EST_EXE2;
				end
				else begin
					op_sel = `ALUOP_MAIN;
					ina_mux = `AMUX_RDAT;
					inb_mux = `BMUX_OPRND;
					cc_mux = `CCMUX_AOP;
					rbus_mux = `RMUX_WR;
					rnum_mux = `RNUM_OPREG;
					rwd_mux = `RWDMUX_OUTA;
					inst_req = 1'b1;
					state_nxt = state_def;
				end
			end	// OPC_BOP
			else begin
				// do nothing
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// AMODE_IMM8,AMODE_IMM16
		else if(amode == `AMODE_REL) begin
			if(opcode[7:4] == `OPC_BCC) begin
				if(cond_ok) begin
					jmp_req = 1'b1;
					op_sel = `ALUOP_ADD16;
					ina_mux = `AMUX_PC;
					inb_mux = `BMUX_OPRND8SE;
					pc_mux = `PCMUX_OUTA;
					if(jmp_ack) begin
						state_nxt = `EST_IDLE;
					end
				end
				else begin
					inst_req = 1'b1;
					state_nxt = state_def;
				end
			end	// OPC_BCC
			else if(opcode == `OPC_BSR) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					tmp_mux = `TMUX_RDAT;
					op_sel = `ALUOP_SUB16;
					ina_mux = `AMUX_RDAT;
					inb_mux = `BMUX_ONE;
					dbus_mux = `DMUX_WR16;
					da_mux = `DAMUX_OUTA;
					dwd_mux = `DWDMUX_PC;
					state_nxt = `EST_EXE3;
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						op_sel = `ALUOP_SUB16;
						ina_mux = `AMUX_TMP;
						inb_mux = `BMUX_TWO;
						rbus_mux = `RMUX_WR;
						rnum_mux = `RNUM_OPREG;
						rwd_mux = `RWDMUX_OUTA;
						state_nxt = `EST_EXE4;
					end
				end
				else begin
					jmp_req = 1'b1;
					op_sel = `ALUOP_ADD16;
					ina_mux = `AMUX_PC;
					inb_mux = `BMUX_OPRND8SE;
					pc_mux = `PCMUX_OUTA;
					if(jmp_ack) begin
						state_nxt = `EST_IDLE;
					end
				end
			end	// OPC_BSR
			else begin
				// do nothing
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end	// AMODE_REL
		// instructions with effective address calc
		else begin
			if(amode == `AMODE_IDX) begin
				if(state == `EST_AM) begin
					op_sel = `ALUOP_ADD16;
					ina_mux = `AMUX_IBASE;
					inb_mux = `BMUX_OPRND8ZE;
					ea_mux = `EAMUX_OUTA;
					state_nxt = `EST_EXE1;
				end
			end
			else if(state == `EST_AM) begin
				// skip to next state
				state_nxt = `EST_EXE1;
			end

			if({opcode[7],opcode[3:0]} == `OPC_BIT) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_AND;
						ina_mux = `AMUX_RDAT;
						inb_mux = `BMUX_DDAT;
						cc_mux = `CCMUX_LOP;
						inst_req = 1'b1;
						state_nxt = state_def;
					end
					else begin
						rbus_mux = `RMUX_RD;
						rnum_mux = `RNUM_OPREG;
					end
				end
			end	// OPC_BIT
			else if(opcode == `OPC_TST) begin
				if(state == `EST_EXE1) begin
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_THRU;
						ina_mux = `AMUX_DDAT;
						cc_mux = `CCMUX_LOP;
						inst_req = 1'b1;
						state_nxt = state_def;
					end
				end
			end	// OPC_TST
			else if({opcode[7:5],opcode[3:0]} == `OPC_CLR) begin
				if(state == `EST_EXE1) begin
					op_sel = `ALUOP_THRU;
					ina_mux = `AMUX_ZERO;
					cc_mux = `CCMUX_AOP2;
					dbus_mux = `DMUX_WR;
					da_mux = `DAMUX_EA;
					dwd_mux = `DWDMUX_OUTA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						state_nxt = state_def;
					end
				end
			end	// OPC_CLR
			else if({opcode[7],opcode[3:0]} == `OPC_CMP) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_SUB;
						ina_mux = `AMUX_RDAT;
						inb_mux = `BMUX_DDAT;
						cc_mux = `CCMUX_AOP2;
						inst_req = 1'b1;
						state_nxt = state_def;
					end
					else begin
						rbus_mux = `RMUX_RD;
						rnum_mux = `RNUM_OPREG;
					end
				end
			end	// OPC_CMP
			else if({opcode[7:6],opcode[3:0]} == `OPC_CMP16) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_SUB;
						ina_mux = `AMUX_RDAT;
						inb_mux = `BMUX_DDAT;
						cc_mux = `CCMUX_LOP;
						inst_req = 1'b1;
						state_nxt = state_def;
					end
					else begin
						rbus_mux = `RMUX_RD;
						rnum_mux = `RNUM_OPREG;
					end
				end
			end	// OPC_CMP16
			else if({opcode[7:5],opcode[3:0]} == `OPC_JMP) begin
				if(state == `EST_EXE1) begin
					jmp_req = 1'b1;
					pc_mux = `PCMUX_EA;
					if(jmp_ack) begin
						state_nxt = `EST_IDLE;
					end
				end
			end	// OPC_JMP
			else if({opcode[7:5],opcode[3:0]} == `OPC_JSR) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					tmp_mux = `TMUX_RDAT;
					op_sel = `ALUOP_SUB16;
					ina_mux = `AMUX_RDAT;
					inb_mux = `BMUX_ONE;
					dbus_mux = `DMUX_WR16;
					da_mux = `DAMUX_OUTA;
					dwd_mux = `DWDMUX_PC;
					state_nxt = `EST_EXE3;
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						op_sel = `ALUOP_SUB16;
						ina_mux = `AMUX_TMP;
						inb_mux = `BMUX_TWO;
						rbus_mux = `RMUX_WR;
						rnum_mux = `RNUM_OPREG;
						rwd_mux = `RWDMUX_OUTA;
						state_nxt = `EST_EXE4;
					end
				end
				else if(state == `EST_EXE4) begin
					jmp_req = 1'b1;
					pc_mux = `PCMUX_EA;
					if(jmp_ack) begin
						state_nxt = `EST_IDLE;
					end
				end
			end	// OPC_JSR
			else if(
				({opcode[7],opcode[3:0]} == `OPC_LD)
				| ({opcode[7],opcode[3:0]} == `OPC_LD16)
			) begin
				if(state == `EST_EXE1) begin
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_THRU;
						ina_mux = `AMUX_DDAT;
						rbus_mux = `RMUX_WR;
						rnum_mux = `RNUM_OPREG;
						rwd_mux = `RWDMUX_OUTA;
						cc_mux = `CCMUX_LOP;
						inst_req = 1'b1;
						state_nxt = state_def;
					end
				end
			end	// OPC_LD
			else if(
				({opcode[7],opcode[3:0]} == `OPC_ST)
				| ({opcode[7],opcode[3:0]} == `OPC_ST16)
			) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					op_sel = `ALUOP_THRU;
					ina_mux = `AMUX_RDAT;
					cc_mux = `CCMUX_LOP;
					dbus_mux = `DMUX_WR;
					da_mux = `DAMUX_EA;
					dwd_mux = `DWDMUX_OUTA;
					state_nxt = `EST_EXE3;
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						state_nxt = state_def;
					end
				end
			end	// OPC_ST,OPC_ST16
			else if({opcode[7:5],opcode[3:0]} == `OPC_TST) begin
				if(state == `EST_EXE1) begin
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_TST;
						ina_mux = `AMUX_DDAT;
						cc_mux = `CCMUX_AOP;
						inst_req = 1'b1;
						state_nxt = state_def;
					end
				end
			end	// OPC_TST
			else if(opcode[7:5] == `OPC_UOP) begin
				if(state == `EST_EXE1) begin
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_MAIN;
						ina_mux = `AMUX_DDAT;
						cc_mux = `CCMUX_AOP;
						dbus_mux = `DMUX_WR;
						da_mux = `DAMUX_EA;
						dwd_mux = `DWDMUX_OUTA;
						state_nxt = `EST_EXE3;
					end
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						state_nxt = state_def;
					end
				end
			end	// OPC_UOP
			else if(opcode[7] == `OPC_BOP) begin
				if(state == `EST_EXE1) begin
					rbus_mux = `RMUX_RD;
					rnum_mux = `RNUM_OPREG;
					dbus_mux = `DMUX_RD;
					da_mux = `DAMUX_EA;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_MAIN;
						ina_mux = `AMUX_RDAT;
						inb_mux = `BMUX_DDAT;
						cc_mux = `CCMUX_AOP;
						rbus_mux = `RMUX_WR;
						rnum_mux = `RNUM_OPREG;
						rwd_mux = `RWDMUX_OUTA;
						inst_req = 1'b1;
						state_nxt = state_def;
					end
					else begin
						rbus_mux = `RMUX_RD;
						rnum_mux = `RNUM_OPREG;
					end
				end
			end	// OPC_BOP
			else begin
				// do nothing
				inst_req = 1'b1;
				state_nxt = state_def;
			end
		end
	end
end	// always comb

endmodule

// End of ac68_exe_ctrl.v
