//
// acia.v
//
// last modified on 081002
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`define	ACIA_RAVG

module acia(
	clk,
	rst,
	addr,
	en,
	wen,
	wdata,
	rdata,
	irq,
	txd,
	rxd,
	adiv_sel
);

parameter	BRG_MAX0 = 16'h003f;
parameter	BRG_MAX1 = 16'h0000;
parameter	BRG_MAX2 = 16'h0000;
parameter	BRG_MAX3 = 16'h0000;

input	clk;
input	rst;
input	addr;
input	en;
input	wen;
input [7:0]	wdata;
output [7:0]	rdata;
output	irq;	// high active
output	txd;
input	rxd;
input [1:0]	adiv_sel;


wire	addr0_en;
wire	stat_ren;
wire	ctrl_wen;
wire	addr1_en;
wire	rdat_ren;
wire	tdat_wen;

reg	rcv_full;
reg	trn_emp;
reg	dcd_r;
reg	cts_r;
reg	frm_err;
reg	ovrn;
reg	pt_err;
wire [7:0]	stat_reg;
reg [1:0]	div_sel;
reg [2:0]	mode;
reg [1:0]	tcon;
reg	rcv_ien;
reg [7:0]	rcv_reg;
reg [7:0]	trn_reg;

wire	mrst;

reg [15:0]	brg_cntr;
reg [15:0]	brg_rld;
wire	scnt0;
reg [2:0]	tdiv_cntr;
wire	tcnt;
reg [2:0]	rdiv_cntr;
wire	rcnt;

wire	rts;

wire	tdre;	// transmit data reg empty
wire	rdrf;	// receive data reg full

reg [3:0]	trn_cntr;
reg [3:0]	trn_cntr_nxt;
reg [3:0]	tcntr_max;
reg [9:0]	tshft_reg;
reg [9:0]	tshft_fmt;
reg	tparity;
wire	tshft_load;
wire	snd_brk;
reg	txd;

reg	rxd_d;
wire	rxd_fall;
`ifdef	ACIA_RAVG
reg [1:0]	rxd_smp;
reg [1:0]	rxd_smp_nxt;
`endif
wire	rxd_avg;
reg [3:0]	rcv_cntr;
reg [3:0]	rcv_cntr_nxt;
reg [3:0]	rcntr_max;
reg [10:0]	rshft_reg;
reg [10:0]	rshft_reg_nxt;
reg	frm_ok;
reg	rparity_ok;
reg [7:0]	rshft_fmt;
reg	rcv_reg_load;
reg	receiving;

wire	trn_ien;
reg	trn_irq;
reg	rcv_irq;


// bus interface
assign	addr0_en = (addr == 1'b0) & en;
assign	stat_ren = addr0_en & ~wen;
assign	ctrl_wen = addr0_en & wen;
assign	addr1_en = (addr == 1'b1) & en;
assign	rdat_ren = addr1_en & ~wen;
assign	tdat_wen = addr1_en & wen;

assign	rdata = ({8{stat_ren}} & stat_reg)
			| ({8{rdat_ren}} & rcv_reg);


//
// status register
//
assign	stat_reg = {
		irq, pt_err, ovrn, frm_err,
		cts_r, dcd_r, tdre, rdrf
	};


//
// control register
//
always @(posedge clk or negedge rst) begin
	if(~rst) begin
		div_sel <= 2'b00;
		mode <= 3'b000;
		tcon <= 2'b00;
		rcv_ien <= 1'b0;
	end
	else if(ctrl_wen) begin
		div_sel <= wdata[1:0];
		mode <= wdata[4:2];
		tcon <= wdata[6:5];
		rcv_ien <= wdata[7];
	end
end

//
// receive data register
//
always @(posedge clk or negedge rst) begin
	if(~rst)
		rcv_reg <= 8'h00;
	else if(rcv_reg_load)
		rcv_reg <= rshft_fmt;
end

//
// transmit data register
//
always @(posedge clk or negedge rst) begin
	if(~rst)
		trn_reg <= 8'h00;
	else if(tdat_wen)
		trn_reg <= wdata;
end

//
// master reset
//
assign	mrst = (div_sel == 2'b11);

//
// baudrate generator
//
always @(posedge clk or negedge rst) begin
	if(~rst)
		brg_cntr <= brg_rld;
	else if(brg_cntr == 16'h0000)
		brg_cntr <= brg_rld;
	else
		brg_cntr <= brg_cntr - 16'h0001;
end

always @(adiv_sel or div_sel) begin
	case(adiv_sel)
	2'b01:		// preset 1
		brg_rld = BRG_MAX1;
	2'b10:		// preset 2
		brg_rld = BRG_MAX2;
	2'b11:		// preset 3
		brg_rld = BRG_MAX3;
	default: begin	// preset 0 scaled by div_sel
		case(div_sel)
		2'b00:		// 1/1
			brg_rld = (BRG_MAX0 >> 6);
		2'b01:		// 1/16
			brg_rld = (BRG_MAX0 >> 2);
		default:	// 1/64 & mrst
			brg_rld = BRG_MAX0;
		endcase
	end
	endcase
end	// always comb

assign	scnt0 = (brg_cntr == 16'h0000);

always @(posedge clk or negedge rst) begin
	if(~rst)
		tdiv_cntr <= 3'b000;
	else if(scnt0)
		tdiv_cntr <= tdiv_cntr + 3'b001;
end

assign	tcnt = (tdiv_cntr == 3'b111) & scnt0;

always @(posedge clk or negedge rst) begin
	if(~rst)
		rdiv_cntr <= 3'b000;
	else if(rxd_fall) begin
		rdiv_cntr <= 3'b011;
	end
	else if(scnt0)
		rdiv_cntr <= rdiv_cntr + 3'b001;
end

assign	rcnt = (rdiv_cntr == 3'b111) & scnt0;

//
// flow control
//

// data carrier detect
always @(posedge clk or negedge rst) begin
	if(~rst)
		dcd_r <= 1'b0;
	else if(mrst)
		dcd_r <= 1'b0;
end

// clear to send
always @(posedge clk or negedge rst) begin
	if(~rst)
		cts_r <= 1'b0;
end

// request to send
assign	rts = (tcon == 2'b10);

//
// buffer control
//
always @(posedge clk or negedge rst) begin
	if(~rst)
		trn_emp <= 1'b1;
	else if(tdat_wen)
		trn_emp <= 1'b0;
	else if(tshft_load)
		trn_emp <= 1'b1;
end

assign	tdre = trn_emp & (~cts_r);

always @(posedge clk or negedge rst) begin
	if(~rst)
		rcv_full <= 1'b0;
	else if(mrst)
		rcv_full <= 1'b0;
	else if(rcv_reg_load)
		rcv_full <= 1'b1;
	else if(rdat_ren)
		rcv_full <= 1'b0;
end

assign	rdrf = rcv_full;


//
// transmit control
//
always @(posedge clk or negedge rst) begin
	if(~rst)
		trn_cntr <= 4'h0;
	else
		trn_cntr <= trn_cntr_nxt;
end

always @(
	trn_cntr or tcnt or tshft_load or tcntr_max
) begin
	if(tshft_load)
		trn_cntr_nxt = tcntr_max;
	else if(trn_cntr == 4'h0)
		trn_cntr_nxt = 4'h0;
	else if(tcnt)
		trn_cntr_nxt = trn_cntr - 4'h1;
	else
		trn_cntr_nxt = trn_cntr;
end	// always comb

assign	tshft_load = (~tdre) & (trn_cntr == 4'd0) & tcnt;

always @(posedge clk or negedge rst) begin
	if(~rst)
		tshft_reg <= 10'h000;
	else if(tshft_load)
		tshft_reg <= tshft_fmt;
	else if(tcnt)
		tshft_reg <= {1'b1,tshft_reg[9:1]};
end

always @(mode or trn_reg) begin
	if(mode[2]) begin	// 8bit
		if(mode[0])	// odd
			tparity = ~(^trn_reg[7:0]);
		else		// even
			tparity = ^trn_reg[7:0];
	end
	else begin			// 7bit
		if(mode[0])	// odd
			tparity = ~(^trn_reg[6:0]);
		else		// even
			tparity = ^trn_reg[6:0];
	end
end	// always comb

always @(mode or trn_reg or tparity) begin
	case(mode)
	3'b000,			// 7bit,even,2stop
	3'b001: begin	// 7bit,odd,2stop
		tcntr_max = 4'd11;
		rcntr_max = 4'd10;
		tshft_fmt = {1'b1,tparity,trn_reg[6:0],1'b0};
	end
	3'b010,			// 7bit,even,1stop
	3'b011: begin	// 7bit,odd,1stop
		tcntr_max = 4'd10;
		rcntr_max = 4'd10;
		tshft_fmt = {1'b1,tparity,trn_reg[6:0],1'b0};
	end
	3'b100: begin	// 8bit,no,2stop
		tcntr_max = 4'd11;
		rcntr_max = 4'd10;
		tshft_fmt = {1'b1,trn_reg[7:0],1'b0};
	end
	3'b101: begin	// 8bit,no,1stop
		tcntr_max = 4'd10;
		rcntr_max = 4'd10;
		tshft_fmt = {1'b1,trn_reg[7:0],1'b0};
	end
	default: begin	// 8bit,even/odd,1stop
		tcntr_max = 4'd11;
		rcntr_max = 4'd11;
		tshft_fmt = {tparity,trn_reg[7:0],1'b0};
	end
	endcase
end	// always comb

assign	snd_brk = (tcon == 2'b11);

always @(snd_brk or tshft_reg or trn_cntr) begin
	if(snd_brk)
		txd = 1'b0;
	else if(trn_cntr != 4'd0)
		txd = tshft_reg[0];
	else
		txd = 1'b1;
end	// always comb

//
// receive control
//
always @(posedge clk or negedge rst) begin
	if(~rst)
		rxd_d <= 1'b1;
	else if(scnt0)
		rxd_d <= rxd;
end

assign	rxd_fall = rxd_d & ~rxd & ~receiving;

always @(posedge clk or negedge rst) begin
	if(~rst)
		receiving <= 1'b0;
	else if(rxd_fall)
		receiving <= 1'b1;
	else if(rcv_reg_load)
		receiving <= 1'b0;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		rcv_cntr <= 4'h0;
	else
		rcv_cntr <= rcv_cntr_nxt;
end

always @(
	rcv_cntr or rcnt or rxd_fall or rcntr_max
) begin
	if(rxd_fall)
		rcv_cntr_nxt = rcntr_max;
	else if(rcv_cntr == 4'h0)
		rcv_cntr_nxt = 4'h0;
	else if(rcnt)
		rcv_cntr_nxt = rcv_cntr - 4'h1;
	else
		rcv_cntr_nxt = rcv_cntr;
end	// always comb

`ifdef	ACIA_RAVG
always @(posedge clk or negedge rst) begin
	if(~rst)
		rxd_smp <= 2'b00;
	else if(scnt0)
		rxd_smp <= rxd_smp_nxt;
end

always @(rdiv_cntr or rxd or rxd_smp) begin
	if(rdiv_cntr == 3'b100)
		rxd_smp_nxt = 2'b00;
	else if((rdiv_cntr == 3'b101)
		| (rdiv_cntr == 3'b110)
		| (rdiv_cntr == 3'b111)
	) begin
		if(rxd)
			rxd_smp_nxt = rxd_smp + 2'b01;
		else
			rxd_smp_nxt = rxd_smp;
	end
	else
		rxd_smp_nxt = rxd_smp;
end

assign	rxd_avg = rxd_smp_nxt[1];
`else	// ACIA_RAVG
assign	rxd_avg = rxd;
`endif	// ACIA_RAVG

always @(posedge clk or negedge rst) begin
	if(~rst)
		rshft_reg <= 11'h000;
	else
		rshft_reg <= rshft_reg_nxt;
end

always @(
	rcnt or rcv_cntr or rxd_avg or rshft_reg
) begin
	if(rcnt & rcv_cntr != 4'h0)
		rshft_reg_nxt = {rxd_avg,rshft_reg[10:1]};
	else
		rshft_reg_nxt = rshft_reg;
end	// always comb

always @(mode or rshft_reg_nxt) begin
	case(mode)
	3'b000, 3'b001:	// 7bit,e/o parity,2stop
		rshft_fmt = {1'b0, rshft_reg_nxt[8:2]};
	3'b010, 3'b011:	// 7bit,e/o parity,1stop
		rshft_fmt = {1'b0, rshft_reg_nxt[8:2]};
	3'b101:			// 8bit,no parity,1stop
		rshft_fmt = rshft_reg_nxt[9:2];
	3'b110, 3'b111:	// 8bit,e/o parity,1stop
		rshft_fmt = rshft_reg_nxt[8:1];
	default:		// 3'b100 8bit,no parity,2stop
		rshft_fmt = rshft_reg_nxt[9:2];
	endcase
end	// always comb

always @(mode or rshft_reg_nxt) begin
	case(mode)
	3'b000, 3'b001:	// 7bit,e/o parity,2stop
		frm_ok = (~rshft_reg_nxt[1]) & rshft_reg_nxt[10];	// check 1st stopbit
	3'b010, 3'b011:	// 7bit,e/o parity,1stop
		frm_ok = (~rshft_reg_nxt[1]) & rshft_reg_nxt[10];
	3'b101:			// 8bit,no parity,1stop
		frm_ok = (~rshft_reg_nxt[1]) & rshft_reg_nxt[10];
	3'b110, 3'b111:	// 8bit,e/o parity,1stop
		frm_ok = (~rshft_reg_nxt[0]) & rshft_reg_nxt[10];
	default:		// 3'b100 8bit,no parity,2stop
		frm_ok = (~rshft_reg_nxt[1]) & rshft_reg_nxt[10];	// check 1st stopbit
	endcase
end	// always comb

always @(mode or rshft_reg_nxt) begin
	case(mode[2:1])
	2'b00:	// 7bit,e/o parity 2stop
		rparity_ok = (mode[0] == (^rshft_reg_nxt[9:2]));
	2'b01:	// 7bit,e/o parity 1stop
		rparity_ok = (mode[0] == (^rshft_reg_nxt[9:2]));
	2'b11:	// 8bit,e/o parity 1stop
		rparity_ok = (mode[0] == (^rshft_reg_nxt[9:1]));
	default:
		rparity_ok = 1'b1;
	endcase
end	// always comb

always @(
	receiving or rdrf or rdat_ren or rcv_cntr_nxt
) begin
	if(receiving) begin
		if(~rdrf & rcv_cntr_nxt == 4'h0)
			rcv_reg_load = 1'b1;
		else if(rdrf & rdat_ren & rcv_cntr_nxt == 4'h0)
			rcv_reg_load = 1'b1;
		else
			rcv_reg_load = 1'b0;
	end
	else
		rcv_reg_load = 1'b0;
end	// always comb

//
// error detection
//
always @(posedge clk or negedge rst) begin
	if(~rst)
		frm_err <= 1'b0;
	else if(rcv_reg_load)
		frm_err <= ~frm_ok;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		ovrn <= 1'b0;
	else if(mrst)
		ovrn <= 1'b0;
	else if(rcv_reg_load & rdrf)
		ovrn <= 1'b1;
	else if(rdat_ren)
		ovrn <= 1'b0;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		pt_err <= 1'b0;
	else if(rcv_reg_load)
		pt_err <= ~rparity_ok;
end

//
// interrupt request
//
assign	trn_ien = (tcon == 2'b01);

always @(trn_ien or tdre) begin
	if(trn_ien)
		trn_irq = tdre;
	else
		trn_irq = 1'b0;
end	// always comb

always @(rcv_ien or rdrf or dcd_r) begin
	if(rcv_ien)
		rcv_irq = rdrf | dcd_r;
	else
		rcv_irq = 1'b0;
end	// always comb

assign	irq = (trn_irq | rcv_irq);

endmodule

// End of acia.v
