//
// ac09_fetch.v
//
// ac09 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac09_defs.v"

`define	FST_LEN		3
`define	FST_IDLE	3'b000
`define	FST_OPCODE1	3'b001
`define	FST_OPCODE2	3'b010
`define	FST_OPRND1	3'b011
`define	FST_OPRND2	3'b100

module ac09_fetch(
	clk,
	rst,
	inst_rdy,
	inst_ack,
	pc_nxt,
	jmp_req,
	jmp_ack,
	new_pc,
	opcode,
	main_opr,
	cond,
	opreg,
	amode,
	postbyte,
	oprnd,
	calc_ea,
	paddr,
	pen,
	prdata,
	pack
);

input	clk;
input	rst;
output	inst_rdy;
input	inst_ack;
output [15:0]	pc_nxt;
input	jmp_req;
output	jmp_ack;
input [15:0]	new_pc;
output [`OPC_LEN - 1:0]	opcode;
output [`ALUOP_LEN - 1:0]	main_opr;
output [3:0]	cond;
output [`REG_LEN - 1:0]	opreg;
output [`AMODE_LEN - 1:0]	amode;
output [7:0]	postbyte;
output [15:0]	oprnd;
output	calc_ea;
output [15:0]	paddr;
output	pen;
input [7:0]	prdata;
input	pack;

reg [`FST_LEN - 1:0]	state;
reg [`FST_LEN - 1:0]	state_nxt;

reg [15:0]	pc;
reg [15:0]	pc_nxt;
wire [15:0]	pc_plus_1;
wire [15:0]	pc_plus_2;

reg	inst_rdy;
reg	jmp_ack;

wire	opc_stat;
reg [`OPC_LEN - 1:0]	opcode_r;
reg [`OPC_LEN - 1:0]	opcode_nxt;

reg [`ALUOP_LEN - 1:0]	main_opr_r;
reg [`ALUOP_LEN - 1:0]	main_opr_nxt;

reg [3:0]	cond_r;
reg [3:0]	cond_nxt;

reg [`REG_LEN - 1:0]	opreg_r;
reg [`REG_LEN - 1:0]	opreg_nxt;

reg [`AMODE_LEN - 1:0]	amode_r;
reg [`AMODE_LEN - 1:0]	amode_nxt;

reg [7:0]	postbyte_r;
reg [7:0]	postbyte_nxt;

reg [15:0]	oprnd_r;
reg [15:0]	oprnd_nxt;

reg	calc_ea_r;
reg	calc_ea_nxt;

reg	code_pg;	// 0 - page2, 1 - page3
reg	code_pg_nxt;

reg [1:0]	req_dat;
reg [1:0]	req_dat_nxt;

wire [15:0]	paddr_i;
reg	pen_i;
reg	psize_i;
wire [15:0]	prdata_i;
wire	pack_i;


always @(posedge clk or negedge rst) begin
	if(~rst)
		state <= `FST_IDLE;
	else
		state <= state_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		pc <= 16'h0000;
	else
		pc <= pc_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		opcode_r <= `OPC_NOP;
		main_opr_r <= `ALUOP_NONE;
		cond_r <= 4'b0000;
		opreg_r <= `REG_ACCD;
		amode_r <= `AMODE_NONE;
		postbyte_r <= 8'h00;
		oprnd_r <= 16'h0000;
		calc_ea_r <= 1'b0;
	end
	else begin
		opcode_r <= opcode_nxt;
		main_opr_r <= main_opr_nxt;
		cond_r <= cond_nxt;
		opreg_r <= opreg_nxt;
		amode_r <= amode_nxt;
		postbyte_r <= postbyte_nxt;
		oprnd_r <= oprnd_nxt;
		calc_ea_r <= calc_ea_nxt;
	end
end

assign	opc_stat = (state == `FST_OPCODE1) | (state == `FST_OPCODE2);
assign	opcode = opc_stat ? opcode_nxt : opcode_r;
assign	main_opr = opc_stat ? main_opr_nxt : main_opr_r;
assign	cond = opc_stat ? cond_nxt : cond_r;
assign	opreg = opc_stat ? opreg_nxt : opreg_r;
assign	amode = opc_stat ? amode_nxt : amode_r;
assign	postbyte = postbyte_nxt;
assign	oprnd = oprnd_nxt;
assign	calc_ea = opc_stat ? calc_ea_nxt : calc_ea_r;

always @(posedge clk or negedge rst) begin
	if(~rst)
		code_pg <= 1'b0;
	else
		code_pg <= code_pg_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		req_dat <= 2'b00;
	else
		req_dat <= req_dat_nxt;
end

assign	pc_plus_1 = pc + 16'h0001;
assign	pc_plus_2 = pc + 16'h0002;

always @(
	state or pc or pc_plus_1 or pc_plus_2 or new_pc
	or code_pg or opcode_r or main_opr_r or cond_r
	or opreg_r or amode_r
	or postbyte_r or oprnd_r or calc_ea_r
	or req_dat or req_dat_nxt or prdata_i
	or inst_ack or jmp_req or pack_i
) begin
	pc_nxt = pc;
	pen_i = 1'b0;
	psize_i = 1'b0;	// 8bit
	req_dat_nxt = req_dat;
	code_pg_nxt = code_pg;
	opcode_nxt = opcode_r;
	main_opr_nxt = main_opr_r;
	cond_nxt = cond_r;
	opreg_nxt = opreg_r;
	amode_nxt = amode_r;
	postbyte_nxt = postbyte_r;
	oprnd_nxt = oprnd_r;
	calc_ea_nxt = calc_ea_r;
	inst_rdy = 1'b0;
	state_nxt = state;

	if(state == `FST_IDLE) begin
		if(jmp_req) begin
			pc_nxt = new_pc;
			jmp_ack = 1'b1;
			pen_i = 1'b1;
			state_nxt = `FST_OPCODE1;
		end
		else begin
			jmp_ack = 1'b0;
			inst_rdy = 1'b1;
			if(inst_ack) begin
				pen_i = 1'b1;
				state_nxt = `FST_OPCODE1;
			end
		end	// jmp_req - else
	end	// FST_IDLE
	else begin
		pen_i = 1'b0;
//		req_dat_nxt = 2'b00;

		if(pack_i) begin
			if(jmp_req) begin
				pc_nxt = new_pc;
				jmp_ack = 1'b1;
				pen_i = 1'b1;
				state_nxt = `FST_OPCODE1;
			end
			else begin
				jmp_ack = 1'b0;
				if(state == `FST_OPCODE1) begin
//					opcode_nxt = `OPC_NOP;
					opcode_nxt = `OPC_UNDEF;
					amode_nxt = `AMODE_NONE;
					calc_ea_nxt = 1'b0;
					req_dat_nxt = 2'b00;
					if(prdata_i == 8'h10) begin			// page2
						code_pg_nxt = 1'b0;
						pen_i = 1'b1;
						state_nxt = `FST_OPCODE2;
					end
					else if(prdata_i == 8'h11) begin	// page3
						code_pg_nxt = 1'b1;
						pen_i = 1'b1;
						state_nxt = `FST_OPCODE2;
					end
					else begin
						pen_i = 1'b1;
						state_nxt = `FST_OPRND1;
						if(prdata_i == 8'h1c) begin	// ANDCC
							opcode_nxt = `OPC_LOPCC;
							main_opr_nxt = `ALUOP_AND;
							opreg_nxt = `REG_CCR;
							amode_nxt = `AMODE_IMM;
						end
						else if(prdata_i == 8'h1a) begin	// ORCC
							opcode_nxt = `OPC_LOPCC;
							main_opr_nxt = `ALUOP_OR;
							opreg_nxt = `REG_CCR;
							amode_nxt = `AMODE_IMM;
						end
						else if(prdata_i == 8'h3c) begin	// CWAI
							opcode_nxt = `OPC_CWAI;
							main_opr_nxt = `ALUOP_AND;
							opreg_nxt = `REG_CCR;
							amode_nxt = `AMODE_IMM;
						end
						else if(prdata_i == 8'h16) begin	// LBRA
							opcode_nxt = `OPC_BCC;
							cond_nxt = 4'b0000;	// always
							amode_nxt = `AMODE_REL16;
						end
						else if(prdata_i == 8'h17) begin	// LBSR
							opcode_nxt = `OPC_BSR;
							opreg_nxt = `REG_SSP;
							amode_nxt = `AMODE_REL16;
						end
						else if(prdata_i == 8'h8d) begin	// BSR
							opcode_nxt = `OPC_BSR;
							opreg_nxt = `REG_SSP;
							amode_nxt = `AMODE_REL8;
						end
						else if(prdata_i[7:2] == 6'b0011_00) begin	// LEA
							case(prdata_i[1:0])
							2'b00: begin	// LEAX
								opcode_nxt = `OPC_LEA;
								opreg_nxt = `REG_IX;
							end
							2'b01: begin	// LEAY
								opcode_nxt = `OPC_LEA;
								opreg_nxt = `REG_IY;
							end
							2'b10: begin	// LEAS
								opcode_nxt = `OPC_LEAS;
								opreg_nxt = `REG_SSP;
							end
							default: begin	// 2'b11 LEAU
								opcode_nxt = `OPC_LEAS;
								opreg_nxt = `REG_USP;
							end
							endcase
							amode_nxt = `AMODE_IDX;
						end
						else if(prdata_i[7:2] == 6'b0011_01) begin	// PSHS,PULS,PSHU,PULU
							opcode_nxt = `OPC_UNDEF;
							case(prdata_i[1:0])
							2'b00: begin	// PSHS
								opcode_nxt = `OPC_PUSH;
								opreg_nxt = `REG_SSP;
							end
							2'b01: begin	// PULS
								opcode_nxt = `OPC_PULL;
								opreg_nxt = `REG_SSP;
							end
							2'b10: begin	// PSHU
								opcode_nxt = `OPC_PUSH;
								opreg_nxt = `REG_USP;
							end
							default: begin	// 2'b11 PULU
								opcode_nxt = `OPC_PULL;
								opreg_nxt = `REG_USP;
							end
							endcase
							amode_nxt = `AMODE_PB;	// postbyte
						end
						else if(prdata_i == 8'h12) begin	// NOP
							opcode_nxt = `OPC_NOP;
						end
						else if(prdata_i == 8'h1e) begin	// EXG
							opcode_nxt = `OPC_EXG;
							amode_nxt = `AMODE_IMM;
							opreg_nxt = `REG_ACCA;	// 8bit dummy
						end
						else if(prdata_i == 8'h1f) begin	// TFR
							opcode_nxt = `OPC_TFR;
							amode_nxt = `AMODE_IMM;
							opreg_nxt = `REG_ACCA;	// 8bit dummy
						end
						else begin
							case({prdata_i[7:6],prdata_i[3:0]})
							6'b00_0000: begin
								case(prdata_i[5:4])
								2'b00: begin	// NEG dir
									opcode_nxt = `OPC_NEG;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b00_0011: begin
								case(prdata_i[5:4])
								2'b00: begin	// COM dir
									opcode_nxt = `OPC_COM;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								2'b01: begin	// SYNC
									opcode_nxt = `OPC_SYNC;
								end
								endcase
							end
							6'b00_0100: begin
								case(prdata_i[5:4])
								2'b00: begin	// LSR dir
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_LSR;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b00_0110: begin
								case(prdata_i[5:4])
								2'b00: begin	// ROR dir
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_ROR;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b00_0111: begin
								case(prdata_i[5:4])
								2'b00: begin	// ASR dir
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_ASR;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b00_1000: begin
								case(prdata_i[5:4])
								2'b00: begin	// LSL dir
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_LSL;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b00_1001: begin
								case(prdata_i[5:4])
								2'b00: begin	// ROL dir
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_ROL;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								2'b01: begin	// DAA
									opcode_nxt = `OPC_DAA;
									opreg_nxt = `REG_ACCA;
								end
								2'b11: begin	// RTS
									opcode_nxt = `OPC_RTS;
									opreg_nxt = `REG_SSP;
								end
								endcase
							end
							6'b00_1010: begin
								case(prdata_i[5:4])
								2'b00: begin	// DEC dir
									opcode_nxt = `OPC_INDC;
									main_opr_nxt = `ALUOP_SUB;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								2'b11: begin	// ABX
									opcode_nxt = `OPC_ABX;
									opreg_nxt = `REG_IX;
								end
								endcase
							end
							6'b00_1011: begin
								case(prdata_i[5:4])
								2'b11: begin	// RTI
									opcode_nxt = `OPC_RTI;
									opreg_nxt = `REG_SSP;
								end
								endcase
							end
							6'b00_1100: begin
								case(prdata_i[5:4])
								2'b00: begin	// INC dir
									opcode_nxt = `OPC_INDC;
									main_opr_nxt = `ALUOP_ADD;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b00_1101: begin
								case(prdata_i[5:4])
								2'b00: begin	// TST dir
									opcode_nxt = `OPC_TST;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								2'b01: begin	// SEX
									opcode_nxt = `OPC_SEX;
									opreg_nxt = `REG_ACCD;
								end
								2'b11: begin	// MUL
									opcode_nxt = `OPC_MUL;
									opreg_nxt = `REG_ACCD;
								end
								endcase
							end
							6'b00_1111: begin
								case(prdata_i[5:4])
								2'b00: begin	// CLR dir
									opcode_nxt = `OPC_CLR;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								2'b11: begin	// SWI
									opcode_nxt = `OPC_SWI;
								end
								endcase
							end
							6'b00_1110: begin
								case(prdata_i[5:4])
								2'b00: begin	// JMP dir
									opcode_nxt = `OPC_JMP;
								end
								endcase
							end
							6'b01_0000: begin
								case(prdata_i[5:4])
								2'b00: begin	// NEGA
									opcode_nxt = `OPC_NEGR;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// NEGB
									opcode_nxt = `OPC_NEGR;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// NEG ind,ext
									opcode_nxt = `OPC_NEG;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_0011: begin
								case(prdata_i[5:4])
								2'b00: begin	// COMA
									opcode_nxt = `OPC_COMR;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// COMB
									opcode_nxt = `OPC_COMR;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// COM ind,ext
									opcode_nxt = `OPC_COM;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_0100: begin
								case(prdata_i[5:4])
								2'b00: begin	// LSRA
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_LSR;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// LSRB
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_LSR;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// LSR ind,ext
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_LSR;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_0110: begin
								case(prdata_i[5:4])
								2'b00: begin	// RORA
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_ROR;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// RORB
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_ROR;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// ROR ind,ext
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_ROR;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_0111: begin
								case(prdata_i[5:4])
								2'b00: begin	// ASRA
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_ASR;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// ASRB
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_ASR;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// ASR ind,ext
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_ASR;
									opreg_nxt = `REG_ACCB;	// 8bit dummy
								end
								endcase
							end
							6'b01_1000: begin
								case(prdata_i[5:4])
								2'b00: begin	// LSLA
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_LSL;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// LSLB
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_LSL;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// LSL ind,ext
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_LSL;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_1001: begin
								case(prdata_i[5:4])
								2'b00: begin	// ROLA
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_ROL;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// ROLB
									opcode_nxt = `OPC_SHFR;
									main_opr_nxt = `ALUOP_ROL;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// ROL ind,ext
									opcode_nxt = `OPC_SHF;
									main_opr_nxt = `ALUOP_ROL;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_1010: begin
								case(prdata_i[5:4])
								2'b00: begin	// DECA
									opcode_nxt = `OPC_INDCR;
									main_opr_nxt = `ALUOP_SUB;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// DECB
									opcode_nxt = `OPC_INDCR;
									main_opr_nxt = `ALUOP_SUB;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// DEC ind,ext
									opcode_nxt = `OPC_INDC;
									main_opr_nxt = `ALUOP_SUB;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_1100: begin
								case(prdata_i[5:4])
								2'b00: begin	// INCA
									opcode_nxt = `OPC_INDCR;
									main_opr_nxt = `ALUOP_ADD;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// INCB
									opcode_nxt = `OPC_INDCR;
									main_opr_nxt = `ALUOP_ADD;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// INC ind,ext
									opcode_nxt = `OPC_INDC;
									main_opr_nxt = `ALUOP_ADD;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_1101: begin
								case(prdata_i[5:4])
								2'b00: begin	// TSTA
									opcode_nxt = `OPC_TSTR;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// TSTB
									opcode_nxt = `OPC_TSTR;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// TST ind,ext
									opcode_nxt = `OPC_TST;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b01_1110: begin	// JMP
								case(prdata_i[5:4])
								default: begin	// JMP ind,ext
									opcode_nxt = `OPC_JMP;
								end
								endcase
							end
							6'b01_1111: begin
								case(prdata_i[5:4])
								2'b00: begin	// CLRA
									opcode_nxt = `OPC_CLRR;
									opreg_nxt = `REG_ACCA;
								end
								2'b01: begin	// CLRB
									opcode_nxt = `OPC_CLRR;
									opreg_nxt = `REG_ACCB;
								end
								default: begin	// CLR ind,ext
									opcode_nxt = `OPC_CLR;
									opreg_nxt = `REG_ACCA;	// 8bit dummy
								end
								endcase
							end
							6'b10_0000: begin	// SUBA
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_SUB;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_0001: begin	// CMPA
								opcode_nxt = `OPC_CMP;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_0010: begin	// SBCA
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_SBC;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_0011: begin	// SUBD
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_SUB;
								opreg_nxt = `REG_ACCD;
							end
							6'b10_0100: begin	// ANDA
								opcode_nxt = `OPC_LOP;
								main_opr_nxt = `ALUOP_AND;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_0101: begin	// BITA
								opcode_nxt = `OPC_BIT;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_0110: begin	// LDA
								opcode_nxt = `OPC_LD;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_0111: begin	// STA
								opcode_nxt = `OPC_ST;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_1000: begin	// EORA
								opcode_nxt = `OPC_LOP;
								main_opr_nxt = `ALUOP_EOR;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_1001: begin	// ADCA
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_ADC;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_1010: begin	// ORA
								opcode_nxt = `OPC_LOP;
								main_opr_nxt = `ALUOP_OR;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_1011: begin	// ADDA
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_ADD;
								opreg_nxt = `REG_ACCA;
							end
							6'b10_1100: begin	// CMPX
								opcode_nxt = `OPC_CMP;
								opreg_nxt = `REG_IX;
							end
							6'b10_1101: begin	// JSR 
								opcode_nxt = `OPC_JSR;
								opreg_nxt = `REG_SSP;
							end
							6'b10_1110: begin	// LDX
								opcode_nxt = `OPC_LD;
								opreg_nxt = `REG_IX;
							end
							6'b10_1111: begin	// STX
								opcode_nxt = `OPC_ST;
								opreg_nxt = `REG_IX;
							end
							6'b11_0000: begin	// SUBB
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_SUB;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_0001: begin	// CMPB
								opcode_nxt = `OPC_CMP;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_0010: begin	// SBCB
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_SBC;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_0011: begin	// ADDD
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_ADD;
								opreg_nxt = `REG_ACCD;
							end
							6'b11_0100: begin	// ANDB
								opcode_nxt = `OPC_LOP;
								main_opr_nxt = `ALUOP_AND;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_0101: begin	// BITB
								opcode_nxt = `OPC_BIT;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_0110: begin	// LDB
								opcode_nxt = `OPC_LD;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_0111: begin	// STB
								opcode_nxt = `OPC_ST;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_1000: begin	// EORB
								opcode_nxt = `OPC_LOP;
								main_opr_nxt = `ALUOP_EOR;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_1001: begin	// ADCB
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_ADC;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_1010: begin	// ORB
								opcode_nxt = `OPC_LOP;
								main_opr_nxt = `ALUOP_OR;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_1011: begin	// ADDB
								opcode_nxt = `OPC_AOP;
								main_opr_nxt = `ALUOP_ADD;
								opreg_nxt = `REG_ACCB;
							end
							6'b11_1100: begin	// LDD
								opcode_nxt = `OPC_LD;
								opreg_nxt = `REG_ACCD;
							end
							6'b11_1101: begin	// STD
								opcode_nxt = `OPC_ST;
								opreg_nxt = `REG_ACCD;
							end
							6'b11_1110: begin	// LDU
								opcode_nxt = `OPC_LD;
								opreg_nxt = `REG_USP;
							end
							6'b11_1111: begin	// STU
								opcode_nxt = `OPC_ST;
								opreg_nxt = `REG_USP;
							end
							endcase

							case(prdata_i[7:4])
							4'b0010: begin	// short relative
								amode_nxt = `AMODE_REL8;
								opcode_nxt = `OPC_BCC;
								cond_nxt = prdata_i[3:0];
							end
							4'b1000,
							4'b1100:	// immediate
								amode_nxt = `AMODE_IMM;
							4'b0000,
							4'b1001,
							4'b1101:	// direct
								amode_nxt = `AMODE_DIR;
							4'b0110,
							4'b1010,
							4'b1110:	// index
								amode_nxt = `AMODE_IDX;
							4'b0111,
							4'b1011,
							4'b1111:	// extended
								amode_nxt = `AMODE_EXT;
							default:
								amode_nxt = `AMODE_NONE;
							endcase
						end
						case(amode_nxt)
						`AMODE_IMM: begin
							pen_i = 1'b1;
							if(opreg_nxt[3])
								psize_i = 1'b0;	// 8bit
							else
								psize_i = 1'b1;	// 16bit
						end
						`AMODE_DIR: begin
							pen_i = 1'b1;
						end
						`AMODE_IDX: begin
							pen_i = 1'b1;
							calc_ea_nxt = 1'b1;
						end
						`AMODE_EXT: begin
							pen_i = 1'b1;
							psize_i = 1'b1;	// 16bit
						end
						`AMODE_REL8: begin
							pen_i = 1'b1;
						end
						`AMODE_REL16: begin
							pen_i = 1'b1;
							psize_i = 1'b1;	// 16bit
						end
						`AMODE_PB: begin
							pen_i = 1'b1;
						end
						`AMODE_NONE: begin
							inst_rdy = 1'b1;
							if(inst_ack) begin
								pen_i = 1'b1;
								state_nxt = `FST_OPCODE1;
							end
							else begin
								state_nxt = `FST_IDLE;
							end
						end
						endcase
					end
					pc_nxt = pc_plus_1;
				end	// FST_OPCODE1
				else if(state == `FST_OPCODE2) begin
					opcode_nxt = `OPC_NOP;
					amode_nxt = `AMODE_NONE;
					if(code_pg == 1'b0) begin	// page2
						if(prdata_i[7:4] == 4'b0010) begin	// LBcc
							amode_nxt = `AMODE_REL16;
							opcode_nxt = `OPC_BCC;
							cond_nxt = prdata_i[3:0];
							state_nxt = `FST_OPRND1;
						end
						else if(prdata_i == 8'h3f) begin	// SWI2
							opcode_nxt = `OPC_SWI2;
							inst_rdy = 1'b1;
							if(inst_ack) begin
								pen_i = 1'b1;
								state_nxt = `FST_OPCODE1;
							end
							else begin
								state_nxt = `FST_IDLE;
							end
						end
						else begin	// prdata_i[7] == 1
							case({prdata_i[6],prdata_i[3:0]})
							5'b0_0011: begin	// CMPD
								opcode_nxt = `OPC_CMP;
								opreg_nxt = `REG_ACCD;
							end
							5'b0_1100: begin	// CMPY
								opcode_nxt = `OPC_CMP;
								opreg_nxt = `REG_IY;
							end
							5'b0_1110: begin	// LDY
								opcode_nxt = `OPC_LD;
								opreg_nxt = `REG_IY;
							end
							5'b0_1111: begin	// STY
								opcode_nxt = `OPC_ST;
								opreg_nxt = `REG_IY;
							end
							5'b1_1110: begin	// LDS
								opcode_nxt = `OPC_LD;
								opreg_nxt = `REG_SSP;
							end
							5'b1_1111: begin	// STS
								opcode_nxt = `OPC_ST;
								opreg_nxt = `REG_SSP;
							end
							endcase
							case(prdata_i[5:4])
							2'b00:		// imm
								amode_nxt = `AMODE_IMM;
							2'b01:		// direct
								amode_nxt = `AMODE_DIR;
							2'b10:		// index
								amode_nxt = `AMODE_IDX;
							default:	// 2'b11 ext
								amode_nxt = `AMODE_EXT;
							endcase
							state_nxt = `FST_OPRND1;
						end
					end	// page2
					else begin					// page3
						if(prdata_i == 8'h3f) begin	// SWI3
							opcode_nxt = `OPC_SWI3;
							inst_rdy = 1'b1;
							if(inst_ack) begin
								pen_i = 1'b1;
								state_nxt = `FST_OPCODE1;
							end
							else begin
								state_nxt = `FST_IDLE;
							end
						end
						else begin	// prdata_i[7] == 1
							case({prdata_i[6],prdata_i[3:0]})
							5'b0_0011: begin	// CMPU
								opcode_nxt = `OPC_CMP;
								opreg_nxt = `REG_USP;
							end
							5'b0_1100: begin	// CMPS
								opcode_nxt = `OPC_CMP;
								opreg_nxt = `REG_SSP;
							end
							endcase
							case(prdata_i[5:4])
							2'b00:		// imm
								amode_nxt = `AMODE_IMM;
							2'b01:		// direct
								amode_nxt = `AMODE_DIR;
							2'b10:		// index
								amode_nxt = `AMODE_IDX;
							default:	// 2'b11 ext
								amode_nxt = `AMODE_EXT;
							endcase
							state_nxt = `FST_OPRND1;
						end
					end	// page3
					case(amode_nxt)
					`AMODE_IMM: begin
						pen_i = 1'b1;
						psize_i = 1'b1;	// 16bit
					end
					`AMODE_DIR: begin
						pen_i = 1'b1;
					end
					`AMODE_IDX: begin
						pen_i = 1'b1;
						calc_ea_nxt = 1'b1;
					end
					`AMODE_EXT: begin
						pen_i = 1'b1;
						psize_i = 1'b1;	// 16bit
					end
					`AMODE_REL16: begin
						pen_i = 1'b1;
						psize_i = 1'b1;	// 16bit
					end
					endcase
					pc_nxt = pc_plus_1;
				end	// FST_OPCODE2
				else if(state == `FST_OPRND1) begin
					req_dat_nxt = 2'b00;
					if(amode_r == `AMODE_IDX) begin
						postbyte_nxt = prdata_i[7:0];
						if({prdata_i[7],prdata_i[3],prdata_i[1:0]} == 4'b1100)
							req_dat_nxt = 2'b01;	// 8bit
						else if({prdata_i[7],prdata_i[3],prdata_i[1:0]} == 4'b1101)
							req_dat_nxt = 2'b10;	// 16bit
						else if(prdata_i == 8'h9f)	// extended indirect
							req_dat_nxt = 2'b10;
					end
					else if(amode_r == `AMODE_PB) begin
						postbyte_nxt = prdata_i[7:0];
					end
					else begin
						oprnd_nxt = prdata_i;
					end
					case(amode_r)
					`AMODE_IMM: begin
						if(opreg_r[3]) pc_nxt = pc_plus_1;
						else pc_nxt = pc_plus_2;
					end
					`AMODE_EXT: begin
						pc_nxt = pc_plus_2;
					end
					`AMODE_REL16: begin
						pc_nxt = pc_plus_2;
					end
					default:
						pc_nxt = pc_plus_1;
					endcase

					if(req_dat_nxt == 2'b01) begin	// 8bit
						pen_i = 1'b1;
						psize_i = 1'b0;
						state_nxt = `FST_OPRND2;
					end
					else if(req_dat_nxt == 2'b10) begin	// 16bit
						pen_i = 1'b1;
						psize_i = 1'b1;
						state_nxt = `FST_OPRND2;
					end
					else begin	// req_dat_nxt = 2'b00
						inst_rdy = 1'b1;
						if(inst_ack) begin
							pen_i = 1'b1;
							state_nxt = `FST_OPCODE1;
						end
						else begin
							state_nxt = `FST_IDLE;
						end
					end
				end	// FST_OPRND1
				else begin	// FST_OPRND2
					oprnd_nxt = prdata_i;
					if(req_dat == 2'b01)
						pc_nxt = pc_plus_1;
					else	// req_dat == 2'b10
						pc_nxt = pc_plus_2;
					inst_rdy = 1'b1;
					if(inst_ack) begin
						pen_i = 1'b1;
						state_nxt = `FST_OPCODE1;
					end
					else begin
						state_nxt = `FST_IDLE;
					end
				end
			end	// jmp_req - else
		end
		else begin
			jmp_ack = 1'b0;
		end	// pack_i - else
	end	// FST_IDLE - else
end	// always comb

assign	paddr_i = pc_nxt;

ac09_mconr mconr0(
	.clk(clk),
	.rst(rst),
	.addr(paddr_i),
	.en(pen_i),
	.size(psize_i),
	.rdata(prdata_i),
	.ack(pack_i),
	.addr8(paddr),
	.en8(pen),
	.rdata8(prdata),
	.ack8(pack)
);

endmodule

// End of ac09_fetch.v
