//
// vidconv.
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`define	VST_LEN		3
`define	VST_IDLE	3'b000
`define	VST_CLS0	3'b001
`define	VST_CLS1	3'b010
`define	VST_SCRL0	3'b011
`define	VST_SCRL1	3'b100
`define	VST_SCRL2	3'b101

`define	CUROP_LEN	3
`define	CUROP_NULL	3'b000
`define	CUROP_HOME	3'b001
`define	CUROP_BTM	3'b010
`define	CUROP_INC	3'b011
`define	CUROP_NL	3'b100

`define	YOP_LEN		2
`define	YOP_NULL	2'b00
`define	YOP_INIT	2'b01
`define	YOP_INC		2'b10

`define	VMUX_LEN	2
`define	VMUX_NULL	2'b00
`define	VMUX_RD		2'b01
`define	VMUX_WRBL	2'b10
`define	VMUX_WRDT	2'b11

module vidcon(
	clk,
	rst,
	vram_addr,
	vram_en,
	vram_wen,
	vram_wdata,
	vram_rdata,
	crom_addr,
	crom_en,
	crom_rdata,
	cls_req,
	disp_req,
	disp_ack,
	disp_data,
	xcnt,
	ycnt,
	r_in,
	g_in,
	b_in,
	hblnk,
	vblnk
);

input	clk;
input	rst;
output [10:0]	vram_addr;
output	vram_en;
output	vram_wen;
output [7:0]	vram_wdata;
input [7:0]	vram_rdata;
output [8:0]	crom_addr;
output	crom_en;
input [7:0]	crom_rdata;
input	cls_req;
input	disp_req;
output	disp_ack;
input [6:0]	disp_data;
input [9:0]	xcnt;
input [8:0]	ycnt;
output [3:0]	r_in;
output [3:0]	g_in;
output [3:0]	b_in;
input	hblnk;
input	vblnk;

reg [`VST_LEN - 1:0]	state;
reg [`VST_LEN - 1:0]	state_nxt;

reg [`CUROP_LEN - 1:0]	cur_op;
reg [`YOP_LEN - 1:0]	yoff_op;
reg [`VMUX_LEN - 1:0]	vram_mux;
reg	cls;
wire	vram_read;
wire	disp_req_i;
wire	disp_cr;
reg	disp_ack;
reg	disp_ack_i;

reg [8:4]	y_offst;
reg [8:4]	y_offst_nxt;
wire [8:4]	ycnt_mod;
wire [8:4]	cur_y_mod;

reg [9:4]	cur_x;
reg [8:4]	cur_y;
wire [8:4]	cur_y_plus1;
reg [9:4]	cur_x_nxt;
reg [8:4]	cur_y_nxt;
wire	cur_left;
wire	cur_out;
wire	cur_det;
reg	cur_det_d;

reg [3:0]	xcnt_d;
reg [3:0]	xcnt_dd;
reg [3:0]	ycnt_d;
reg [10:0]	vram_addr;
reg	vram_en;
reg	vram_wen;
reg [7:0]	vram_wdata;
reg [7:0]	vram_rdata_r;
reg [7:0]	vram_rdata_i;
reg	pixel_dat;
reg	blnk_d;
wire	blnk;


always @(posedge clk or negedge rst) begin
	if(~rst) begin
		xcnt_d <= 4'b0000;
		xcnt_dd <= 4'b0000;
		ycnt_d <= 4'b0000;
	end
	else begin
		xcnt_d <= xcnt[3:0];
		xcnt_dd <= xcnt_d;
		ycnt_d <= ycnt[3:0];
	end
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		state <= `VST_CLS0;
	else
		state <= state_nxt;
end

assign	vram_read = ~blnk & (xcnt[3:0] == 4'b0000);
assign	disp_req_i = disp_req & ~disp_ack;
assign	disp_cr = disp_req_i & (disp_data == 7'h0d);

always @(
	state or cur_out or cur_left
	or disp_req_i or disp_cr or cls_req
	or vram_read
) begin
	state_nxt = state;
	cur_op = `CUROP_NULL;
	yoff_op = `YOP_NULL;
	vram_mux = `VMUX_NULL;
	cls = 1'b0;
	disp_ack_i = 1'b0;

	if(state == `VST_CLS0) begin
		cur_op = `CUROP_HOME;
		yoff_op = `YOP_INIT;
		state_nxt = `VST_CLS1;
	end	// VST_CLS0
	else if(state == `VST_CLS1) begin
		if(cur_out) begin
			cur_op = `CUROP_HOME;
			state_nxt = `VST_IDLE;
		end
		else begin
			cur_op = `CUROP_INC;
			vram_mux = `VMUX_WRBL;	// write blank
			cls = 1'b1;
		end
	end	// VST_CLS1
	else begin
		if(vram_read) begin
			vram_mux = `VMUX_RD;
		end

		if(state == `VST_SCRL0) begin
			if(cur_out) begin
				state_nxt = `VST_SCRL1;
			end
			else begin
				disp_ack_i = 1'b1;
				state_nxt = `VST_IDLE;
			end
		end	// VST_SCRL0
		else if(state == `VST_SCRL1) begin
			if(~vram_read) begin
				cur_op = `CUROP_INC;
				vram_mux = `VMUX_WRBL;	// write blank
				state_nxt = `VST_SCRL2;
			end
		end	// VST_SCRL1
		else if(state == `VST_SCRL2) begin
			if(cur_left) begin
				cur_op = `CUROP_BTM;
				yoff_op = `YOP_INC;
				disp_ack_i = 1'b1;
				state_nxt = `VST_IDLE;
			end
			else begin
				if(~vram_read) begin
					cur_op = `CUROP_INC;
					vram_mux = `VMUX_WRBL;	// write blank
				end
			end
		end	// VST_SCRL2
		else if(state == `VST_IDLE) begin
			if(cls_req) begin
				state_nxt = `VST_CLS0;
			end
			else if(disp_cr) begin
				cur_op = `CUROP_NL;
				state_nxt = `VST_SCRL0;
			end
			else if(disp_req_i) begin
				if(~vram_read) begin
					cur_op = `CUROP_INC;
					vram_mux = `VMUX_WRDT;
					state_nxt = `VST_SCRL0;
				end
			end
		end
	end
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		disp_ack <= 1'b0;
	else if(disp_ack_i)
		disp_ack <= 1'b1;
	else if(~disp_req)
		disp_ack <= 1'b0;
end

assign	ycnt_mod = ycnt[8:4] + y_offst;
assign	cur_y_mod = cur_y + y_offst;

always @(
	vram_mux or xcnt or ycnt_mod
	or cur_x or cur_y_mod or disp_data
) begin
	vram_addr = 11'h000;
	vram_en = 1'b0;
	vram_wen = 1'b0;
	vram_wdata = 8'h00;

	case(vram_mux)
	`VMUX_RD: begin
		vram_addr = {ycnt_mod,xcnt[9:4]};
		vram_en = 1'b1;
	end
	`VMUX_WRBL: begin
		vram_addr = {cur_y_mod,cur_x};
		vram_en = 1'b1;
		vram_wen = 1'b1;
		vram_wdata = 8'h20;	// space
	end
	`VMUX_WRDT: begin
		vram_addr = {cur_y_mod,cur_x};
		vram_en = 1'b1;
		vram_wen = 1'b1;
		vram_wdata = {2'b00,disp_data[5:0]};
	end
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		blnk_d <= 1'b1;
	else
		blnk_d <= blnk;
end

assign	blnk = hblnk | vblnk | cls;

always @(posedge clk or negedge rst) begin
	if(~rst)
		y_offst <= 5'd0;
	else
		y_offst <= y_offst_nxt;
end

always @(y_offst or yoff_op) begin
	case(yoff_op)
	`YOP_INIT:
		y_offst_nxt = 5'd0;
	`YOP_INC:
		y_offst_nxt = y_offst + 5'd1;
	default:
		y_offst_nxt = y_offst;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		cur_x <= 6'd0;
		cur_y <= 5'd0;
	end
	else begin
		cur_x <= cur_x_nxt;
		cur_y <= cur_y_nxt;
	end
end

assign	cur_y_plus1 = cur_y + 5'd1;

always @(cur_x or cur_y or cur_y_plus1 or cur_op) begin
	case(cur_op)
	`CUROP_HOME: begin
		cur_x_nxt = 6'd0;
		cur_y_nxt = 5'd0;
	end
	`CUROP_BTM: begin
		cur_x_nxt = 6'd0;
		cur_y_nxt = 5'd23;
	end
	`CUROP_INC: begin
		cur_x_nxt = (cur_x == 6'd39) ? 6'd0 : (cur_x + 6'd1);
		cur_y_nxt = (cur_x == 6'd39) ? cur_y_plus1 : cur_y;
	end
	`CUROP_NL: begin
		cur_x_nxt = 6'd0;
		cur_y_nxt = cur_y_plus1;
	end
	default: begin
		cur_x_nxt = cur_x;
		cur_y_nxt = cur_y;
	end
	endcase
end	// always comb

assign	cur_left = (cur_x == 6'd0);
assign	cur_out = (cur_y == 5'd24);

assign	cur_det = (xcnt[9:4] == cur_x) & (ycnt[8:4] == cur_y);

always @(posedge clk or negedge rst) begin
	if(~rst)
		cur_det_d <= 1'b0;
	else
		cur_det_d <= cur_det;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		vram_rdata_r <= 8'h20;	// space
	else if(xcnt_d == 4'b0000)
		vram_rdata_r <= vram_rdata;
end

always @(
	blnk_d or cur_det_d
	or xcnt_d or vram_rdata or vram_rdata_r
) begin
	if(blnk_d)
		vram_rdata_i = 8'h20;	// space
	else if(cur_det_d)
		vram_rdata_i = 8'h1f;	// underscore
	else if(xcnt_d == 4'b0000)
		vram_rdata_i = vram_rdata;
	else
		vram_rdata_i = vram_rdata_r;
end	// always comb

assign	crom_addr = {vram_rdata_i[5:0],ycnt_d[3:1]};
assign	crom_en = 1'b1;

always @(crom_rdata or xcnt_dd) begin
	case(xcnt_dd[3:1])
	3'b000:	pixel_dat = crom_rdata[7];
	3'b001:	pixel_dat = crom_rdata[6];
	3'b010:	pixel_dat = crom_rdata[5];
	3'b011:	pixel_dat = crom_rdata[4];
	3'b100:	pixel_dat = crom_rdata[3];
	3'b101:	pixel_dat = crom_rdata[2];
	3'b110:	pixel_dat = crom_rdata[1];
	default:pixel_dat = crom_rdata[0];	// 3'b111
	endcase
end	// always comb

assign	r_in = {4{pixel_dat}};
assign	g_in = {4{pixel_dat}};
assign	b_in = {4{pixel_dat}};

endmodule

// End of vidcon.v
