//
// ps2kbd.v
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

module ps2kbd(
	clk,
	rst,
	trn_req,
	trn_dat,
	trn_emp,
	scl_o,
	sda_o
);

parameter	PS2_RLD = 12'h13b;

input	clk;
input	rst;
input	trn_req;
input [9:0]	trn_dat;
output	trn_emp;
output	scl_o;
output	sda_o;

reg [11:0]	brg_cntr;
wire	sclk;
reg [2:0]	tdiv_cntr;
wire	tcnt;
reg [3:0]	trn_cntr;
reg [3:0]	trn_cntr_nxt;
reg [1:0]	trn_seq;
reg [1:0]	trn_seq_nxt;
reg	trn_emp;
reg [9:0]	trn_dat_r;
wire [9:0]	trn_dat_i;
wire	tshft_load;
reg [9:0]	tshft_reg;
wire	tparity;


always @(posedge clk or negedge rst) begin
	if(~rst)
		brg_cntr <= PS2_RLD;
	else if(brg_cntr == 12'h000)
		brg_cntr <= PS2_RLD;
	else
		brg_cntr <= brg_cntr - 12'h001;
end

assign	sclk = (brg_cntr == 12'h000);

always @(posedge clk or negedge rst) begin
	if(~rst)
		tdiv_cntr <= 3'd0;
	else if(sclk)
		tdiv_cntr <= tdiv_cntr + 3'd1;
end

assign	tcnt = (tdiv_cntr == 3'b111) & sclk;

always @(posedge clk or negedge rst) begin
	if(~rst)
		trn_cntr <= 4'd0;
	else
		trn_cntr <= trn_cntr_nxt;
end

always @(trn_cntr or tcnt or tshft_load) begin
	if(tshft_load)
		trn_cntr_nxt = 4'd11;
	else if(trn_cntr == 4'd0)
		trn_cntr_nxt = trn_cntr;
	else if(tcnt)
		trn_cntr_nxt = trn_cntr - 4'd1;
	else
		trn_cntr_nxt = trn_cntr;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		trn_seq <= 2'b00;
	else
		trn_seq <= trn_seq_nxt;
end

always @(
	trn_seq or tshft_load or trn_dat_i
) begin
	if(tshft_load & (trn_seq == 2'b00)) begin
		trn_seq_nxt = trn_dat_i[9:8];
	end
	else if(trn_seq == 2'b00) begin
		trn_seq_nxt = 2'b00;
	end
	else if(tshft_load) begin
		if(trn_seq[0])	// key code extension 0xe0
			trn_seq_nxt = {trn_seq[1],1'b0};
		else	// trn_seq[1] == 1'b1	// key break
			trn_seq_nxt = 2'b00;
	end
	else begin
		trn_seq_nxt = trn_seq;
	end
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		trn_emp <= 1'b1;
	else if(trn_req)
		trn_emp <= 1'b0;
	else if(tshft_load & (trn_seq_nxt == 2'b00))
		trn_emp <= 1'b1;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		trn_dat_r <= 10'h000;
	else if(trn_req)
		trn_dat_r <= trn_dat;
end

assign	trn_dat_i = (trn_req) ? trn_dat : trn_dat_r;

assign	tshft_load = (~trn_emp) & (trn_cntr == 4'd0) & tcnt;

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		tshft_reg <= 10'h3ff;
	end
	else if(tshft_load) begin
		if(trn_seq_nxt[0])
			tshft_reg <= {1'b0,8'he0,1'b0};	// tparity = 0
		else if(trn_seq_nxt[1])
			tshft_reg <= {1'b1,8'hf0,1'b0};	// tparity = 1
		else
			tshft_reg <= {tparity,trn_dat_i[7:0],1'b0};
	end
	else if(tcnt) begin
		tshft_reg <= {1'b1,tshft_reg[9:1]};
	end
end

assign	tparity = ~(^trn_dat_i[7:0]);

assign	scl_o = (trn_cntr != 4'd0) ? ~(tdiv_cntr[2]) : 1'b1;
assign	sda_o = (trn_cntr != 4'd0) ? tshft_reg[0] : 1'b1;

endmodule

// End of ps2kbd.v
