//
// ac6502_exe_ctrl.v
//
// ac6502 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac6502_cfg.v"
`include "ac6502_defs.v"

`define	EXE_STATE_LEN	4
`define	EST_IDLE	4'b0000
`define	EST_EXE1	4'b0001
`define	EST_EXE2	4'b0010
`define	EST_EXE3	4'b0011
`define	EST_EXE4	4'b0100
`define	EST_EXE5	4'b0101
`define	EST_EXE6	4'b0110
`define	EST_EXE7	4'b0111
`define	EST_AM1		4'b1001
`define	EST_AM2		4'b1010
`define	EST_AM3		4'b1011

`define	EST_AMEXE1	(is_ind ? `EST_AM1 : `EST_EXE1)

`define	EXST_LEN	2
`define	EXST_NONE	2'b00
`define	EXST_RST	2'b01
`define	EXST_IRQ	2'b10
`define	EXST_NMI	2'b11

module ac6502_exe_ctrl(
	clk,
	rst,
	irq,
	nmi,
	inst1,
	inst1_r,
	inst_rdy,
	inst_ack,
	jmp_req,
	jmp_ack,
	pc_mux,
	cpc_mux,
	acc_mux,
	ix_mux,
	iy_mux,
	sp_mux,
	psr_mux,
	bus_mux,
	ba_mux,
	ea_mux,
	ea_sav,
	wd_mux,
	tmp_mux,
	tmp16_mux,
	op_sel,
	ina_mux,
	inb_mux,
	upd_flg,
	cond,
	ctrue,
	dec_bit,
	imsk_bit,
	ack
);

input	clk;
input	rst;
input	irq;	// high active
input	nmi;	// high active (rising edge)
input [7:0]	inst1;
input [7:0]	inst1_r;
input	inst_rdy;
output	inst_ack;
output	jmp_req;
input	jmp_ack;
output [`PCMUX_LEN - 1:0]	pc_mux;
output [`CPCMUX_LEN - 1:0]	cpc_mux;
output [`ACCMUX_LEN - 1:0]	acc_mux;
output [`IXMUX_LEN - 1:0]	ix_mux;
output [`IYMUX_LEN - 1:0]	iy_mux;
output [`SPMUX_LEN - 1:0]	sp_mux;
output [`PSRMUX_LEN - 1:0]	psr_mux;
output [`BSMUX_LEN - 1:0]	bus_mux;
output [`BAMUX_LEN - 1:0]	ba_mux;
output [`EAMUX_LEN - 1:0]	ea_mux;
output	ea_sav;
output [`WDMUX_LEN - 1:0]	wd_mux;
output [`TMUX_LEN - 1:0]	tmp_mux;
output [`T16MUX_LEN - 1:0]	tmp16_mux;
output [`ALUOP_LEN - 1:0]	op_sel;
output [`AMUX_LEN - 1:0]	ina_mux;
output [`BMUX_LEN - 1:0]	inb_mux;
output [3:0]	upd_flg;	// [z, n, ov, cy]
output [`COND_LEN - 1:0]	cond;
input	ctrue;
input	dec_bit;
input	imsk_bit;
input	ack;


reg [`EXE_STATE_LEN - 1:0]	state;
reg [`EXE_STATE_LEN - 1:0]	state_nxt;

reg [`EXST_LEN - 1:0]	ex_state;
reg [`EXST_LEN - 1:0]	ex_state_nxt;

reg	inst_req;

reg	jmp_req;
reg	jmp_fetch;
reg	jmp_fetch_nxt;
reg [`PCMUX_LEN - 1:0]	pc_mux;
reg [`CPCMUX_LEN - 1:0]	cpc_mux;
reg [`ACCMUX_LEN - 1:0]	acc_mux;
reg [`IXMUX_LEN - 1:0]	ix_mux;
reg [`IYMUX_LEN - 1:0]	iy_mux;
reg [`SPMUX_LEN - 1:0]	sp_mux;
reg [`PSRMUX_LEN - 1:0]	psr_mux;
reg [`BSMUX_LEN - 1:0]	bus_mux;
reg [`BAMUX_LEN - 1:0]	ba_mux;
reg [`EAMUX_LEN - 1:0]	ea_mux;
reg	ea_sav;
reg [`WDMUX_LEN - 1:0]	wd_mux;
reg [`TMUX_LEN - 1:0]	tmp_mux;
reg [`T16MUX_LEN - 1:0]	tmp16_mux;
reg [`ALUOP_LEN - 1:0]	op_sel;
reg [`AMUX_LEN - 1:0]	ina_mux;
reg [`BMUX_LEN - 1:0]	inb_mux;
reg [3:0]	upd_flg;
reg [`COND_LEN - 1:0]	cond;

reg	irq_i;
reg	nmi_i;
reg	nmi_d;
wire	nmi_det;

wire	is_ind;


always @(posedge clk or negedge rst) begin
	if(~rst)
		state <= `EST_IDLE;
	else
		state <= state_nxt;
end	// always

always @(posedge clk or negedge rst) begin
	if(~rst)
		ex_state <= `EXST_RST;
	else
		ex_state <= ex_state_nxt;
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		jmp_fetch <= 1'b0;
	else
		jmp_fetch <= jmp_fetch_nxt;
end

always @(jmp_fetch or inst_ack or jmp_ack) begin
	if(inst_ack)
		jmp_fetch_nxt = 1'b0;
	else if(jmp_ack)
		jmp_fetch_nxt = 1'b1;
	else
		jmp_fetch_nxt = jmp_fetch;
end	// always comb

always @(jmp_ack or inst_ack or jmp_fetch) begin
	if(jmp_ack) cpc_mux = `CPCMUX_PCN;
	else begin
		if(jmp_fetch) cpc_mux = `CPCMUX_NULL;
		else begin
			if(inst_ack) cpc_mux = `CPCMUX_PCR;
			else cpc_mux = `CPCMUX_NULL;
		end
	end
end	// always comb

// masked irq signal (high active)
always @(irq or imsk_bit) begin
	irq_i = (irq & ~imsk_bit);
end	// always comb

// detect rising edge (high active)
always @(posedge clk or negedge rst) begin
	if(~rst)
		nmi_d <= 1'b0;
	else
		nmi_d <= nmi;
end

assign	nmi_det = (nmi & ~nmi_d);

always @(posedge clk or negedge rst) begin
	if(~rst)
		nmi_i <= 1'b0;
	else if(nmi_det)
		nmi_i <= 1'b1;
	else if(ex_state == `EXST_NMI)
		nmi_i <= 1'b0;
end

// indirect mode
assign	is_ind = (inst1[3:0] == 4'b0001) | (inst1 == `OPC_JMP_IND);

assign	inst_ack = inst_req & inst_rdy;

always @(
	state or inst_rdy or inst1_r
	or ex_state or ack or jmp_ack or is_ind
	or ctrue or dec_bit or irq_i or nmi_i
) begin
	jmp_req = 1'b0;
	pc_mux = `PCMUX_ZERO;
	acc_mux = `ACCMUX_NULL;
	ix_mux = `IXMUX_NULL;
	iy_mux = `IYMUX_NULL;
	sp_mux = `SPMUX_NULL;
	psr_mux = `PSRMUX_NULL;
	bus_mux = `BSMUX_NULL;
	ba_mux = `BAMUX_ZERO;
	ea_mux = `EAMUX_NULL;
	ea_sav = 1'b0;
	wd_mux = `WDMUX_ZERO;
	tmp_mux = `TMUX_NULL;
	tmp16_mux = `T16MUX_NULL;
	op_sel = `ALUOP_NULL;
	ina_mux = `AMUX_ZERO;
	inb_mux = `BMUX_ZERO;
	upd_flg = 4'b0000;
	cond = `COND_EQ;
	inst_req = 1'b0;
	ex_state_nxt = ex_state;
	state_nxt = state;

	if(ex_state == `EXST_RST) begin
		if(state == `EST_IDLE) begin
			bus_mux = `BSMUX_RD;
			ba_mux = `BAMUX_RSTL;
			state_nxt = `EST_EXE1;
		end
		else if(state == `EST_EXE1) begin
			if(ack) begin
				tmp16_mux = `T16MUX_RDATL;
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_RSTH;
				state_nxt = `EST_EXE2;
			end
		end
		else if(state == `EST_EXE2) begin
			if(ack) begin
				tmp16_mux = `T16MUX_RDATH;
				state_nxt = `EST_EXE3;
			end
		end
		else begin
			pc_mux = `PCMUX_TMP16;
			jmp_req = 1'b1;
			if(jmp_ack) begin
				ex_state_nxt = `EXST_NONE;
				state_nxt = `EST_IDLE;
			end
		end
	end
	else if(ex_state == `EXST_NMI) begin
		if(state == `EST_EXE1) begin
			bus_mux = `BSMUX_WR;
			ba_mux = `BAMUX_SP;
			wd_mux = `WDMUX_CPCH;
			sp_mux = `SPMUX_DEC;
			state_nxt = `EST_EXE2;
		end
		else if(state == `EST_EXE2) begin
			if(ack) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_CPCL;
				sp_mux = `SPMUX_DEC;
				state_nxt = `EST_EXE3;
			end
		end
		else if(state == `EST_EXE3) begin
			if(ack) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_PSR;
				sp_mux = `SPMUX_DEC;
				state_nxt = `EST_EXE4;
			end
		end
		else if(state == `EST_EXE4) begin
			if(ack) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_NMIL;
				psr_mux = `PSRMUX_IRQ;
				state_nxt = `EST_EXE5;
			end
		end
		else if(state == `EST_EXE5) begin
			if(ack) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_NMIH;
				tmp16_mux = `T16MUX_RDATL;
				state_nxt = `EST_EXE6;
			end
		end
		else if(state == `EST_EXE6) begin
			if(ack) begin
				tmp16_mux = `T16MUX_RDATH;
				state_nxt = `EST_EXE7;
			end
		end
		else if(state == `EST_EXE7) begin
			inst_req = 1'b1;
			pc_mux = `PCMUX_TMP16;
			jmp_req = 1'b1;
			if(jmp_ack) begin
				ex_state_nxt = `EXST_NONE;
				if(inst_rdy) state_nxt = `EST_AMEXE1;
				else state_nxt = `EST_IDLE;
			end
		end
	end
	else if(ex_state == `EXST_IRQ) begin
		if(state == `EST_EXE1) begin
			bus_mux = `BSMUX_WR;
			ba_mux = `BAMUX_SP;
			wd_mux = `WDMUX_CPCH;
			sp_mux = `SPMUX_DEC;
			state_nxt = `EST_EXE2;
		end
		else if(state == `EST_EXE2) begin
			if(ack) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_CPCL;
				sp_mux = `SPMUX_DEC;
				state_nxt = `EST_EXE3;
			end
		end
		else if(state == `EST_EXE3) begin
			if(ack) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_PSR;
				sp_mux = `SPMUX_DEC;
				state_nxt = `EST_EXE4;
			end
		end
		else if(state == `EST_EXE4) begin
			if(ack) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_IRQL;
				psr_mux = `PSRMUX_IRQ;
				state_nxt = `EST_EXE5;
			end
		end
		else if(state == `EST_EXE5) begin
			if(ack) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_IRQH;
				tmp16_mux = `T16MUX_RDATL;
				state_nxt = `EST_EXE6;
			end
		end
		else if(state == `EST_EXE6) begin
			if(ack) begin
				tmp16_mux = `T16MUX_RDATH;
				state_nxt = `EST_EXE7;
			end
		end
		else if(state == `EST_EXE7) begin
			inst_req = 1'b1;
			pc_mux = `PCMUX_TMP16;
			jmp_req = 1'b1;
			if(jmp_ack) begin
				ex_state_nxt = `EXST_NONE;
				if(inst_rdy) state_nxt = `EST_AMEXE1;
				else state_nxt = `EST_IDLE;
			end
		end
	end
	else if(nmi_i & (state == `EST_IDLE | state == `EST_AMEXE1)) begin
		ex_state_nxt = `EXST_NMI;
		state_nxt = `EST_EXE1;
	end
	else if(irq_i & (state == `EST_IDLE | state == `EST_AMEXE1)) begin
		ex_state_nxt = `EXST_IRQ;
		state_nxt = `EST_EXE1;
	end
	else if(state == `EST_IDLE) begin
		inst_req = 1'b1;
		if(inst_rdy) state_nxt = `EST_AMEXE1;
	end
	else begin
		//
		// instructions without memory access
		//
		if(inst1_r == `OPC_ADC_IMM) begin
			inst_req = 1'b1;
			if(dec_bit) op_sel = `ALUOP_ADCD;
			else op_sel = `ALUOP_ADC;
			ina_mux = `AMUX_ACC;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1111;	// update ZNVC
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_AND_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_AND;
			ina_mux = `AMUX_ACC;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1100;	// update ZN..
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_ASL_A) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_ASL;
			ina_mux = `AMUX_ACC;
			upd_flg = 4'b1101;	// update ZN.C
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r[4:0] == `OPC_BCOND) begin
			inst_req = 1'b1;
			cond = inst1_r[7:5];
			if(ctrue) begin
				jmp_req = 1'b1;
				pc_mux = `PCMUX_REL;
				if(jmp_ack) begin
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
			else begin
				if(inst_rdy) state_nxt = `EST_AMEXE1;
				else state_nxt = `EST_IDLE;
			end
		end
		else if(inst1_r == `OPC_BRK) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_BRKH;
				sp_mux = `SPMUX_DEC;
				state_nxt = `EST_EXE2;
			end
			if(state == `EST_EXE2) begin
				if(ack) begin
					bus_mux = `BSMUX_WR;
					ba_mux = `BAMUX_SP;
					wd_mux = `WDMUX_BRKL;
					sp_mux = `SPMUX_DEC;
					state_nxt = `EST_EXE3;
				end
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					bus_mux = `BSMUX_WR;
					ba_mux = `BAMUX_SP;
					wd_mux = `WDMUX_PSR;
					sp_mux = `SPMUX_DEC;
					state_nxt = `EST_EXE4;
				end
			end
			else if(state == `EST_EXE4) begin
				if(ack) begin
					bus_mux = `BSMUX_RD;
					ba_mux = `BAMUX_IRQL;
					psr_mux = `PSRMUX_BRK;
					state_nxt = `EST_EXE5;
				end
			end
			else if(state == `EST_EXE5) begin
				if(ack) begin
					bus_mux = `BSMUX_RD;
					ba_mux = `BAMUX_IRQH;
					tmp16_mux = `T16MUX_RDATL;
					state_nxt = `EST_EXE6;
				end
			end
			else if(state == `EST_EXE6) begin
				if(ack) begin
					tmp16_mux = `T16MUX_RDATH;
					state_nxt = `EST_EXE7;
				end
			end
			else if(state == `EST_EXE7) begin
				inst_req = 1'b1;
				pc_mux = `PCMUX_TMP16;
				jmp_req = 1'b1;
				if(jmp_ack) begin
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
		else if(inst1_r == `OPC_CLC) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_CLFLG;
			upd_flg = 4'b0001;	// update ...C
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_CLD) begin
			inst_req = 1'b1;
			psr_mux = `PSRMUX_CLD;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_CLI) begin
			inst_req = 1'b1;
			psr_mux = `PSRMUX_CLI;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_CLV) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_CLFLG;
			upd_flg = 4'b0010;	// update ..V.
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_CMP_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_SUB;
			ina_mux = `AMUX_ACC;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1101;	// update ZN.C
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_CPX_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_SUB;
			ina_mux = `AMUX_IX;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1101;	// update ZN.C
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_CPY_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_SUB;
			ina_mux = `AMUX_IY;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1101;	// update ZN.C
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_DEX) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_SUB;
			ina_mux = `AMUX_IX;
			inb_mux = `BMUX_ONE;
			upd_flg = 4'b1100;	// update ZN..
			ix_mux = `IXMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_DEY) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_SUB;
			ina_mux = `AMUX_IY;
			inb_mux = `BMUX_ONE;
			upd_flg = 4'b1100;	// update ZN..
			iy_mux = `IYMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_EOR_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_EOR;
			ina_mux = `AMUX_ACC;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1100;	// update ZN..
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_INX) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_ADD;
			ina_mux = `AMUX_IX;
			inb_mux = `BMUX_ONE;
			upd_flg = 4'b1100;	// update ZN..
			ix_mux = `IXMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_INY) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_ADD;
			ina_mux = `AMUX_IY;
			inb_mux = `BMUX_ONE;
			upd_flg = 4'b1100;	// update ZN..
			iy_mux = `IYMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_JMP_ABS) begin
			inst_req = 1'b1;
			pc_mux = `PCMUX_ABS;
			jmp_req = 1'b1;
			if(jmp_ack) begin
				if(inst_rdy) state_nxt = `EST_AMEXE1;
				else state_nxt = `EST_IDLE;
			end
		end
		else if(inst1_r == `OPC_JMP_IND) begin
			if(state == `EST_AM1) begin
				ba_mux = `BAMUX_ABS;
				bus_mux = `BSMUX_RD;
				state_nxt = `EST_AM2;
			end
			else if(state == `EST_AM2) begin
				if(ack) begin
					ea_mux = `EAMUX_ABS_1;
					ba_mux = `BAMUX_EA;
					bus_mux = `BSMUX_RD;
					tmp16_mux = `T16MUX_RDATL;
					state_nxt = `EST_AM3;
				end
			end
			else if(state == `EST_AM3) begin
				if(ack) begin
					tmp16_mux = `T16MUX_RDATH;
					state_nxt = `EST_EXE1;
				end
			end
			else if(state == `EST_EXE1) begin
				inst_req = 1'b1;
				pc_mux = `PCMUX_TMP16;
				jmp_req = 1'b1;
				if(jmp_ack) begin
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
		else if(inst1_r == `OPC_JSR) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_JSRH;
				sp_mux = `SPMUX_DEC;
				state_nxt = `EST_EXE2;
			end
			if(state == `EST_EXE2) begin
				if(ack) begin
					bus_mux = `BSMUX_WR;
					ba_mux = `BAMUX_SP;
					wd_mux = `WDMUX_JSRL;
					sp_mux = `SPMUX_DEC;
					state_nxt = `EST_EXE3;
				end
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					state_nxt = `EST_EXE4;
				end
			end
			else if(state == `EST_EXE4) begin
				inst_req = 1'b1;
				pc_mux = `PCMUX_ABS;
				jmp_req = 1'b1;
				if(jmp_ack) begin
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
		else if(inst1_r == `OPC_LDA_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_INST2;
			upd_flg = 4'b1100;	// update ZN..
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_LDX_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_INST2;
			upd_flg = 4'b1100;	// update ZN..
			ix_mux = `IXMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_LDY_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_INST2;
			upd_flg = 4'b1100;	// update ZN..
			iy_mux = `IYMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_LSR_A) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_LSR;
			ina_mux = `AMUX_ACC;
			upd_flg = 4'b1101;	// update ZN.C
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_NOP) begin
			inst_req = 1'b1;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_ORA_IMM) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_OR;
			ina_mux = `AMUX_ACC;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1100;	// update ZN..
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_PHA) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_ACC;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_SUB;
					ina_mux = `AMUX_SP;
					inb_mux = `BMUX_ONE;
					sp_mux = `SPMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
		else if(inst1_r == `OPC_PHP) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_PSR;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_SUB;
					ina_mux = `AMUX_SP;
					inb_mux = `BMUX_ONE;
					sp_mux = `SPMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
`ifdef WDC_EXT
		else if(inst1_r == `OPC_PHX) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_IX;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_SUB;
					ina_mux = `AMUX_SP;
					inb_mux = `BMUX_ONE;
					sp_mux = `SPMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
`endif // def WDC_EXT
`ifdef WDC_EXT
		else if(inst1_r == `OPC_PHY) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_WR;
				ba_mux = `BAMUX_SP;
				wd_mux = `WDMUX_IY;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_SUB;
					ina_mux = `AMUX_SP;
					inb_mux = `BMUX_ONE;
					sp_mux = `SPMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
`endif // def WDC_EXT
		else if(inst1_r == `OPC_PLA) begin
			if(state == `EST_EXE1) begin
				op_sel = `ALUOP_ADD;
				ina_mux = `AMUX_SP;
				inb_mux = `BMUX_ONE;
				sp_mux = `SPMUX_OUTA;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_SP;
				state_nxt = `EST_EXE3;
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_THR;
					ina_mux = `AMUX_RDAT;
					upd_flg = 4'b1100;	// update ZN..
					acc_mux = `ACCMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
		else if(inst1_r == `OPC_PLP) begin
			if(state == `EST_EXE1) begin
				op_sel = `ALUOP_ADD;
				ina_mux = `AMUX_SP;
				inb_mux = `BMUX_ONE;
				sp_mux = `SPMUX_OUTA;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_SP;
				state_nxt = `EST_EXE3;
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_THR;
					ina_mux = `AMUX_RDAT;
					psr_mux = `PSRMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
`ifdef WDC_EXT
		else if(inst1_r == `OPC_PLX) begin
			if(state == `EST_EXE1) begin
				op_sel = `ALUOP_ADD;
				ina_mux = `AMUX_SP;
				inb_mux = `BMUX_ONE;
				sp_mux = `SPMUX_OUTA;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_SP;
				state_nxt = `EST_EXE3;
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_THR;
					ina_mux = `AMUX_RDAT;
					upd_flg = 4'b1100;	// update ZN..
					ix_mux = `IXMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
`endif // def WDC_EXT
`ifdef WDC_EXT
		else if(inst1_r == `OPC_PLY) begin
			if(state == `EST_EXE1) begin
				op_sel = `ALUOP_ADD;
				ina_mux = `AMUX_SP;
				inb_mux = `BMUX_ONE;
				sp_mux = `SPMUX_OUTA;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				bus_mux = `BSMUX_RD;
				ba_mux = `BAMUX_SP;
				state_nxt = `EST_EXE3;
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					inst_req = 1'b1;
					op_sel = `ALUOP_THR;
					ina_mux = `AMUX_RDAT;
					upd_flg = 4'b1100;	// update ZN..
					iy_mux = `IYMUX_OUTA;
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
`endif // def WDC_EXT
		else if(inst1_r == `OPC_ROL_A) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_ROL;
			ina_mux = `AMUX_ACC;
			upd_flg = 4'b1101;	// update ZN.C
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_ROR_A) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_ROR;
			ina_mux = `AMUX_ACC;
			upd_flg = 4'b1101;	// update ZN.C
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_RTI) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_RD;
				ea_mux = `EAMUX_SP_1;
				ba_mux = `BAMUX_SPEA;
				sp_mux = `SPMUX_INC;
				state_nxt = `EST_EXE2;
			end
			else if(state == `EST_EXE2) begin
				if(ack) begin
					bus_mux = `BSMUX_RD;
					ea_mux = `EAMUX_SP_1;
					ba_mux = `BAMUX_SPEA;
					psr_mux = `PSRMUX_RDAT;
					sp_mux = `SPMUX_INC;
					state_nxt = `EST_EXE3;
				end
			end
			if(state == `EST_EXE3) begin
				if(ack) begin
					bus_mux = `BSMUX_RD;
					ea_mux = `EAMUX_SP_1;
					ba_mux = `BAMUX_SPEA;
					tmp16_mux = `T16MUX_RDATL;
					sp_mux = `SPMUX_INC;
					state_nxt = `EST_EXE4;
				end
			end
			else if(state == `EST_EXE4) begin
				if(ack) begin
					tmp16_mux = `T16MUX_RDATH;
					state_nxt = `EST_EXE5;
				end
			end
			else if(state == `EST_EXE5) begin
				inst_req = 1'b1;
				pc_mux = `PCMUX_TMP16;
				jmp_req = 1'b1;
				if(jmp_ack) begin
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
		else if(inst1_r == `OPC_RTS) begin
			if(state == `EST_EXE1) begin
				bus_mux = `BSMUX_RD;
				ea_mux = `EAMUX_SP_1;
				ba_mux = `BAMUX_SPEA;
				sp_mux = `SPMUX_INC;
				state_nxt = `EST_EXE2;
			end
			if(state == `EST_EXE2) begin
				if(ack) begin
					bus_mux = `BSMUX_RD;
					ea_mux = `EAMUX_SP_1;
					ba_mux = `BAMUX_SPEA;
					tmp16_mux = `T16MUX_RDATL;
					sp_mux = `SPMUX_INC;
					state_nxt = `EST_EXE3;
				end
			end
			else if(state == `EST_EXE3) begin
				if(ack) begin
					tmp16_mux = `T16MUX_RDATH;
					state_nxt = `EST_EXE4;
				end
			end
			else if(state == `EST_EXE4) begin
				inst_req = 1'b1;
				pc_mux = `PCMUX_RTS;
				jmp_req = 1'b1;
				if(jmp_ack) begin
					if(inst_rdy) state_nxt = `EST_AMEXE1;
					else state_nxt = `EST_IDLE;
				end
			end
		end
		else if(inst1_r == `OPC_SBC_IMM) begin
			inst_req = 1'b1;
			if(dec_bit) op_sel = `ALUOP_SBCD;
			else op_sel = `ALUOP_SBC;
			ina_mux = `AMUX_ACC;
			inb_mux = `BMUX_INST2;
			upd_flg = 4'b1111;	// update ZNVC
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_SEC) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_SEFLG;
			upd_flg = 4'b0001;	// update ...C
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_SED) begin
			inst_req = 1'b1;
			psr_mux = `PSRMUX_SED;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_SEI) begin
			inst_req = 1'b1;
			psr_mux = `PSRMUX_SEI;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_TAX) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_ACC;
			upd_flg = 4'b1100;	// update ZN..
			ix_mux = `IXMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_TAY) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_ACC;
			upd_flg = 4'b1100;	// update ZN..
			iy_mux = `IYMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_TSX) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_SP;
			upd_flg = 4'b1100;	// update ZN..
			ix_mux = `IXMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_TXA) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_IX;
			upd_flg = 4'b1100;	// update ZN..
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_TXS) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_IX;
			sp_mux = `SPMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else if(inst1_r == `OPC_TYA) begin
			inst_req = 1'b1;
			op_sel = `ALUOP_THR;
			ina_mux = `AMUX_IY;
			upd_flg = 4'b1100;	// update ZN..
			acc_mux = `ACCMUX_OUTA;
			if(inst_rdy) state_nxt = `EST_AMEXE1;
			else state_nxt = `EST_IDLE;
		end
		else begin
			//
			// instructions with memory access
			//
			// zero page indexed indirect
			if(inst1_r[4:2] == 3'b000) begin
				if(state == `EST_AM1) begin
					ea_mux = `EAMUX_ZPX;
					ba_mux = `BAMUX_ZPEA;
					ea_sav = 1'b1;
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_AM2;
				end
				else if(state == `EST_AM2) begin
					if(ack) begin
						ea_mux = `EAMUX_INCZP;
						ba_mux = `BAMUX_ZPEA;
						bus_mux = `BSMUX_RD;
						tmp16_mux = `T16MUX_RDATL;
						state_nxt = `EST_AM3;
					end
				end
				else if(state == `EST_AM3) begin
					if(ack) begin
						tmp16_mux = `T16MUX_RDATH;
						state_nxt = `EST_EXE1;
					end
				end
				else begin
					ba_mux = `BAMUX_TMP16;
				end
			end
			// zero page indirect indexed with Y
			else if(inst1_r[4:2] == 3'b100) begin
				if(state == `EST_AM1) begin
					ba_mux = `BAMUX_ZP;
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_AM2;
				end
				else if(state == `EST_AM2) begin
					if(ack) begin
						ea_mux = `EAMUX_ZP_1;
						ba_mux = `BAMUX_ZPEA;
						bus_mux = `BSMUX_RD;
						tmp16_mux = `T16MUX_RDATL;
						state_nxt = `EST_AM3;
					end
				end
				else if(state == `EST_AM3) begin
					if(ack) begin
						tmp16_mux = `T16MUX_RDATH;
						state_nxt = `EST_EXE1;
					end
				end
				else begin
					ea_mux = `EAMUX_TMP16Y;
					ba_mux = `BAMUX_EA;
				end
			end
			// zero page addressing
			else if(inst1_r[4:2] == 3'b001) begin
				ba_mux = `BAMUX_ZP;
			end
			// absolute addressing
			else if(inst1_r[4:2] == 3'b011) begin
				ba_mux = `BAMUX_ABS;
			end
			// zero page indexed with X
			else if(inst1_r[4:2] == 3'b101) begin
				ea_mux = `EAMUX_ZPX;
				ba_mux = `BAMUX_ZPEA;
			end
			// absolute indexed with Y
			else if(inst1_r[4:2] == 3'b110) begin
				ea_mux = `EAMUX_ABSY;
				ba_mux = `BAMUX_EA;
			end
			// absolute indexed with X
			else if(inst1_r[4:2] == 3'b111) begin
				ea_mux = `EAMUX_ABSX;
				ba_mux = `BAMUX_EA;
			end
			if({inst1_r[7:5], inst1_r[1:0]} == `OPC_ADC_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						if(dec_bit) op_sel = `ALUOP_ADCD;
						else op_sel = `ALUOP_ADC;
						ina_mux = `AMUX_ACC;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1111;	// update ZNVC
						acc_mux = `ACCMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_AND_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_AND;
						ina_mux = `AMUX_ACC;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1100;	// update ZN..
						acc_mux = `ACCMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_ASL_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_ASL;
						ina_mux = `AMUX_RDAT;
						upd_flg = 4'b1101;	// update ZN.C
						bus_mux = `BSMUX_WR;
						wd_mux = `WDMUX_OUTA;
						state_nxt = `EST_EXE3;
					end
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_BIT_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_BIT;
						ina_mux = `AMUX_ACC;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1110;	// update ZNV.
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_CMP_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_SUB;
						ina_mux = `AMUX_ACC;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1101;	// update ZN.C
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_CPX_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_SUB;
						ina_mux = `AMUX_IX;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1101;	// update ZN.C
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_CPY_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_SUB;
						ina_mux = `AMUX_IY;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1101;	// update ZN.C
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_DEC_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_SUB;
						ina_mux = `AMUX_RDAT;
						inb_mux = `BMUX_ONE;
						upd_flg = 4'b1100;	// update ZN..
						bus_mux = `BSMUX_WR;
						wd_mux = `WDMUX_OUTA;
						state_nxt = `EST_EXE3;
					end
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_EOR_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_EOR;
						ina_mux = `AMUX_ACC;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1100;	// update ZN..
						acc_mux = `ACCMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_INC_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_ADD;
						ina_mux = `AMUX_RDAT;
						inb_mux = `BMUX_ONE;
						upd_flg = 4'b1100;	// update ZN..
						bus_mux = `BSMUX_WR;
						wd_mux = `WDMUX_OUTA;
						state_nxt = `EST_EXE3;
					end
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_LDA_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_THR;
						ina_mux = `AMUX_RDAT;
						upd_flg = 4'b1100;	// update ZN..
						acc_mux = `ACCMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_LDX_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_THR;
						ina_mux = `AMUX_RDAT;
						upd_flg = 4'b1100;	// update ZN..
						ix_mux = `IXMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_LDY_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_THR;
						ina_mux = `AMUX_RDAT;
						upd_flg = 4'b1100;	// update ZN..
						iy_mux = `IYMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_LSR_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_LSR;
						ina_mux = `AMUX_RDAT;
						upd_flg = 4'b1101;	// update ZN.C
						bus_mux = `BSMUX_WR;
						wd_mux = `WDMUX_OUTA;
						state_nxt = `EST_EXE3;
					end
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_ORA_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						op_sel = `ALUOP_OR;
						ina_mux = `AMUX_ACC;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1100;	// update ZN..
						acc_mux = `ACCMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_ROL_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_ROL;
						ina_mux = `AMUX_RDAT;
						upd_flg = 4'b1101;	// update ZN.C
						bus_mux = `BSMUX_WR;
						wd_mux = `WDMUX_OUTA;
						state_nxt = `EST_EXE3;
					end
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_ROR_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						op_sel = `ALUOP_ROR;
						ina_mux = `AMUX_RDAT;
						upd_flg = 4'b1101;	// update ZN.C
						bus_mux = `BSMUX_WR;
						wd_mux = `WDMUX_OUTA;
						state_nxt = `EST_EXE3;
					end
				end
				else if(state == `EST_EXE3) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_SBC_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_RD;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						if(dec_bit) op_sel = `ALUOP_SBCD;
						else op_sel = `ALUOP_SBC;
						ina_mux = `AMUX_ACC;
						inb_mux = `BMUX_RDAT;
						upd_flg = 4'b1111;	// update ZNVC
						acc_mux = `ACCMUX_OUTA;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_STA_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_WR;
					wd_mux = `WDMUX_ACC;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_STX_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_WR;
					wd_mux = `WDMUX_IX;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
			else if({inst1_r[7:5], inst1_r[1:0]} == `OPC_STY_EA) begin
				if(state == `EST_EXE1) begin
					bus_mux = `BSMUX_WR;
					wd_mux = `WDMUX_IY;
					state_nxt = `EST_EXE2;
				end
				else if(state == `EST_EXE2) begin
					if(ack) begin
						inst_req = 1'b1;
						if(inst_rdy) state_nxt = `EST_AMEXE1;
						else state_nxt = `EST_IDLE;
					end
				end
			end
		end
	end
end	// always comb


endmodule

// End of ac6502_exe_ctrl.v
