//
// ac6502_exe_alu.v
//
// ac6502 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac6502_defs.v"

module ac6502_exe_alu(
	op_sel,
	ina,
	inb,
	cy_bit,
	outa,
	outb,
	new_cy,
	new_ov,
	new_neg,
	new_zero
);

input [`ALUOP_LEN - 1:0]	op_sel;
input [7:0]	ina;
input [7:0]	inb;
input	cy_bit;
output [7:0]	outa;
output [7:0]	outb;
output	new_cy;
output	new_ov;
output	new_neg;
output	new_zero;

reg [7:0]	outa;
reg [7:0]	outb;
reg	new_cy;
reg	new_ov;
reg	new_neg;
reg	new_zero;

reg [2:0]	op_adsb;
wire [7:0]	out_adsb;
wire	ncy_adsb;
wire	nov_adsb;
wire	nng_adsb;
wire	nzr_adsb;


always @(
	op_sel or ina or inb or cy_bit
	or out_adsb or ncy_adsb or nov_adsb
	or nng_adsb or nzr_adsb
	or outa or outb
) begin
	outa = 8'h00;
	outb = 8'h00;
	new_cy = cy_bit;
	new_ov = 1'b0;
	new_neg = 1'b0;
	new_zero = 1'b0;
	op_adsb = 3'b000;

	case(op_sel)
	`ALUOP_ADD: begin
		op_adsb = 3'b000;
		outa = out_adsb;
		new_cy = ncy_adsb;
		new_ov = nov_adsb;
		new_neg = nng_adsb;
		new_zero = nzr_adsb;
	end
	`ALUOP_SUB: begin
		op_adsb = 3'b001;
		outa = out_adsb;
		new_cy = ncy_adsb;
		new_ov = nov_adsb;
		new_neg = nng_adsb;
		new_zero = nzr_adsb;
	end
	`ALUOP_ADC: begin
		op_adsb = 3'b010;
		outa = out_adsb;
		new_cy = ncy_adsb;
		new_ov = nov_adsb;
		new_neg = nng_adsb;
		new_zero = nzr_adsb;
	end
	`ALUOP_SBC: begin
		op_adsb = 3'b011;
		outa = out_adsb;
		new_cy = ncy_adsb;
		new_ov = nov_adsb;
		new_neg = nng_adsb;
		new_zero = nzr_adsb;
	end
	`ALUOP_ADCD: begin
		op_adsb = 3'b100;
		outa = out_adsb;
		new_cy = ncy_adsb;
		new_ov = nov_adsb;
		new_neg = nng_adsb;
		new_zero = nzr_adsb;
	end
	`ALUOP_SBCD: begin
		op_adsb = 3'b101;
		outa = out_adsb;
		new_cy = ncy_adsb;
		new_ov = nov_adsb;
		new_neg = nng_adsb;
		new_zero = nzr_adsb;
	end
	`ALUOP_THR: begin
		outa = ina;
		new_neg = ina[7];
		new_zero = ~(| ina);
	end
	`ALUOP_OR: begin
		outa = ina | inb;
		new_neg = outa[7];
		new_zero = ~(| outa);
	end
	`ALUOP_AND: begin
		outa = ina & inb;
		new_neg = outa[7];
		new_zero = ~(| outa);
	end
	`ALUOP_EOR: begin
		outa = ina ^ inb;
		new_neg = outa[7];
		new_zero = ~(| outa);
	end
	`ALUOP_CLFLG: begin
		new_cy = 1'b0;
		new_ov = 1'b0;
		new_neg = 1'b0;
		new_zero = 1'b0;
	end
	`ALUOP_SEFLG: begin
		new_cy = 1'b1;
		new_ov = 1'b1;
		new_neg = 1'b1;
		new_zero = 1'b1;
	end
	`ALUOP_ASL: begin
		new_cy = ina[7];
		outa = {ina[6:0], 1'b0};
		new_neg = outa[7];
		new_zero = ~(| outa);
	end
	`ALUOP_LSR: begin
		new_cy = ina[0];
		outa = {1'b0, ina[7:1]};
		new_neg = 1'b0;
		new_zero = ~(| outa);
	end
	`ALUOP_ROL: begin
		new_cy = ina[7];
		outa = {ina[6:0], cy_bit};
		new_neg = outa[7];
		new_zero = ~(| outa);
	end
	`ALUOP_ROR: begin
		new_cy = ina[0];
		outa = {cy_bit, ina[7:1]};
		new_neg = outa[7];
		new_zero = ~(| outa);
	end
	`ALUOP_BIT: begin
		outa = ina & inb;
		new_ov = inb[6];
		new_neg = inb[7];
		new_zero = ~(| outa);
	end
	endcase
end	// always comb

ac6502_addsub addsub0(
	.op(op_adsb),
	.ina(ina),
	.inb(inb),
	.cy_bit(cy_bit),
	.out(out_adsb),
	.new_cy(ncy_adsb),
	.new_ov(nov_adsb),
	.new_neg(nng_adsb),
	.new_zero(nzr_adsb)
);

endmodule


// End of ac6502_exe_alu.v
