//
// ac68_exe_alu.v
//
// ac68 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac68_defs.v"

module ac68_exe_alu(
	op_sel,
	opsize,
	ina,
	inb,
	h_bit,
	c_bit,
	v_bit,
	outa,
	outb,
	new_h,
	new_n,
	new_z,
	new_v,
	new_c
);

input [`ALUOP_LEN - 1:0]	op_sel;
input	opsize;
input [15:0]	ina;
input [15:0]	inb;
input	h_bit;
input	c_bit;
input	v_bit;
output [15:0]	outa;
output [15:0]	outb;
output	new_h;
output	new_n;
output	new_z;
output	new_v;
output	new_c;

reg [15:0]	outa;
reg [15:0]	outb;

reg	new_h;
reg	new_n;
reg	new_z;
reg	new_v;
reg	new_c;

reg [1:0]	addsub_op;
reg	addsub_opsize;
reg [15:0]	addsub_ina;
reg [15:0]	addsub_inb;
wire [15:0]	addsub_out;
wire	addsub_cout;
wire	addsub_hout;
wire	addsub_vout;

wire	da_lvld;
wire	da_hvld;
reg [7:0]	da_val;
wire	da_new_c;


always @(
	op_sel or opsize or ina or inb
	or c_bit or v_bit or h_bit or new_n
	or addsub_out or addsub_cout or addsub_hout or addsub_vout
	or da_val or da_new_c
) begin
	outa = 16'h0000;
	outb = 16'h0000;
	new_h = h_bit;
	new_v = v_bit;
	new_c = c_bit;
	addsub_op = 2'b00;
	addsub_opsize = opsize;
	addsub_ina = ina;
	addsub_inb = inb;

	case(op_sel)
	`ALUOP_THRU: begin
		outa = ina;
		new_v = 1'b0;
	end
	`ALUOP_ADD: begin
		addsub_op = 2'b00;
		outa = addsub_out;
		new_c = addsub_cout;
		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_SUB: begin
		addsub_op = 2'b10;
		outa = addsub_out;
		new_c = addsub_cout;
//		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_ADD16: begin
		addsub_op = 2'b00;
		addsub_opsize = 1'b1;	// 16bit
		outa = addsub_out;
		new_c = addsub_cout;
		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_SUB16: begin
		addsub_op = 2'b10;
		addsub_opsize = 1'b1;	// 16bit
		outa = addsub_out;
		new_c = addsub_cout;
		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_AND: begin
		outa = ina & inb;
		new_v = 1'b0;
	end
	`ALUOP_EOR: begin
		outa = ina ^ inb;
		new_v = 1'b0;
	end
	`ALUOP_OR: begin
		outa = ina | inb;
		new_v = 1'b0;
	end
	`ALUOP_COM: begin
		outa = ~ina;
		new_c = 1'b1;
	end
	`ALUOP_NEG: begin
		addsub_op = 2'b10;
		addsub_ina = 16'h0000;
		addsub_inb = ina;
		outa = addsub_out;
		new_c = addsub_cout;
//		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_ADC: begin
		addsub_op = 2'b01;
		outa = addsub_out;
		new_c = addsub_cout;
		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_SBC: begin
		addsub_op = 2'b11;
		outa = addsub_out;
		new_c = addsub_cout;
//		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_DAA: begin
		addsub_op = 2'b00;
		addsub_inb = da_val;
		outa = addsub_out;
//		new_c = addsub_cout;
		new_c = da_new_c;
		new_v = addsub_vout;
	end
	`ALUOP_TST: begin
		addsub_op = 2'b10;
		addsub_ina = ina;
		addsub_inb = 16'h0000;
		outa = addsub_out;
		new_c = addsub_cout;
//		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_ASR: begin
		outa = {ina[7],ina[7:1]};
		new_c = ina[0];
		new_v = new_c^new_n;
	end
	`ALUOP_LSR: begin
		outa = {1'b0,ina[7:1]};
		new_c = ina[0];
		new_v = new_c^new_n;
	end
	`ALUOP_ASL: begin
		outa = {ina[6:0],1'b0};
		new_c = ina[7];
		new_v = new_c^new_n;
	end
	`ALUOP_ROR: begin
		outa = {c_bit,ina[7:1]};
		new_c = ina[0];
		new_v = new_c^new_n;
	end
	`ALUOP_ROL: begin
		outa = {ina[6:0],c_bit};
		new_c = ina[7];
		new_v = new_c^new_n;
	end
	`ALUOP_PSH: begin
		addsub_op = 2'b10;
		addsub_opsize = 1'b1;	// 16bit
		addsub_inb = inb[2] ? 16'h0002 : 16'h0001;
		outa = addsub_out;
	end
	`ALUOP_PUL: begin
		addsub_op = 2'b00;
		addsub_opsize = 1'b1;	// 16bit
		addsub_inb = inb[2] ? 16'h0002: 16'h0001;
		outa = addsub_out;
	end
	`ALUOP_INC: begin
		addsub_op = 2'b00;
		addsub_inb = 16'h0001;
		outa = addsub_out;
		new_c = addsub_cout;
		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	`ALUOP_DEC: begin
		addsub_op = 2'b10;
		addsub_inb = 16'h0001;
		outa = addsub_out;
		new_c = addsub_cout;
//		new_h = addsub_hout;
		new_v = addsub_vout;
	end
	endcase
end	// always comb

always @(opsize or outa) begin
	if(opsize)	// 16bit
		new_n = outa[15];
	else		// 8bit
		new_n = outa[7];
end	// always comb

always @(opsize or outa) begin
	if(opsize)	// 16bit
		new_z = ~(|outa);
	else		// 8bit
		new_z = ~(|outa[7:0]);
end	// always comb

assign	da_lvld = (ina[3:0] <= 4'h9);
assign	da_hvld = (ina[7:4] <= 4'h9);

always @(ina or da_lvld or da_hvld or c_bit or h_bit) begin
	if(c_bit) begin
		if(h_bit) da_val = 8'h66;
		else begin
			if(da_lvld) da_val = 8'h60;
			else da_val = 8'h66;
		end
	end
	else begin
		if(h_bit) begin
			if(da_hvld) da_val = 8'h06;
			else da_val = 8'h66;
		end
		else begin
			if(da_lvld) begin
				if(da_hvld) da_val = 8'h00;
				else da_val = 8'h60;
			end
			else begin
				if(ina[7:4] <= 4'h8) da_val = 8'h06;
				else da_val = 8'h66;
			end
		end
	end
end	// always comb

assign	da_new_c = (da_val[7:4] == 4'h6);

ac68_addsub addsub0(
	.op(addsub_op),
	.opsize(addsub_opsize),
	.ina(addsub_ina),
	.inb(addsub_inb),
	.c_bit(c_bit),
	.h_bit(h_bit),
	.out(addsub_out),
	.new_c(addsub_cout),
	.new_h(addsub_hout),
	.new_v(addsub_vout)
);

endmodule

// End of ac68_exe_alu.v
