//
// ac6502_exe_data.v
//
// ac6502 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac6502_defs.v"

module ac6502_exe_data(
	clk,
	rst,
	pc_nxt,
	inst1,
	inst2,
	inst3,
	inst1_r,
	inst_ack,
	new_pc,
	pc_mux,
	cpc_mux,
	acc_mux,
	ix_mux,
	iy_mux,
	sp_mux,
	psr_mux,
	bus_mux,
	ba_mux,
	ea_mux,
	ea_sav,
	wd_mux,
	tmp_mux,
	tmp16_mux,
	ina_mux,
	inb_mux,
	ina,
	inb,
	cy_bit,
	outa,
	outb,
	new_cy,
	new_ov,
	new_neg,
	new_zero,
	upd_flg,
	cond,
	ctrue,
	dec_bit,
	imsk_bit,
	addr,
	wdata,
	rdata,
	en,
	wen
);

input	clk;
input	rst;
input [15:0]	pc_nxt;
input [7:0]	inst1;
input [7:0]	inst2;
input [7:0]	inst3;
output [7:0]	inst1_r;
input	inst_ack;
output [15:0]	new_pc;
input [`PCMUX_LEN - 1:0]	pc_mux;
input [`CPCMUX_LEN - 1:0]	cpc_mux;
input [`ACCMUX_LEN - 1:0]	acc_mux;
input [`IXMUX_LEN - 1:0]	ix_mux;
input [`IYMUX_LEN - 1:0]	iy_mux;
input [`SPMUX_LEN - 1:0]	sp_mux;
input [`PSRMUX_LEN - 1:0]	psr_mux;
input [`BSMUX_LEN - 1:0]	bus_mux;
input [`BAMUX_LEN - 1:0]	ba_mux;
input [`EAMUX_LEN - 1:0]	ea_mux;
input	ea_sav;
input [`WDMUX_LEN - 1:0]	wd_mux;
input [`TMUX_LEN - 1:0]	tmp_mux;
input [`T16MUX_LEN - 1:0]	tmp16_mux;
input [`AMUX_LEN - 1:0]	ina_mux;
input [`BMUX_LEN - 1:0]	inb_mux;
output [7:0]	ina;
output [7:0]	inb;
output	cy_bit;
input [7:0]	outa;
input [7:0]	outb;
input	new_cy;
input	new_ov;
input	new_neg;
input	new_zero;
input [3:0]	upd_flg;	// [z, n, ov, cy]
input [`COND_LEN - 1:0]	cond;
output	ctrue;
output	dec_bit;
output	imsk_bit;
output [15:0]	addr;
output [7:0]	wdata;
input [7:0]	rdata;
output	en;
output	wen;


reg [15:0]	pc_r;
reg [7:0]	inst1_r;
reg [7:0]	inst2_r;
reg [7:0]	inst3_r;

reg [15:0]	new_pc;

reg [15:0]	cur_pc;
reg [15:0]	cur_pc_nxt;

reg [7:0]	acc;
reg [7:0]	acc_nxt;

reg [7:0]	ix;
reg [7:0]	ix_nxt;

reg [7:0]	iy;
reg [7:0]	iy_nxt;

reg [7:0]	sp;
reg [7:0]	sp_nxt;

reg [7:0]	psr;
reg [7:0]	psr_nxt;

reg	ctrue;

// data area interface
reg [15:0]	addr;
reg [7:0]	wdata;
reg	en;
reg	wen;

reg [7:0]	tmp;
reg [7:0]	tmp_nxt;

reg [15:0]	tmp16;
reg [15:0]	tmp16_nxt;

// ALU interface
reg [7:0]	ina;
reg [7:0]	inb;


always @(posedge clk or negedge rst) begin
	if(~rst) begin
		pc_r <= 16'h0000;
		inst1_r <= 8'h00;
		inst2_r <= 8'h00;
		inst3_r <= 8'h00;
	end
	else if(inst_ack) begin
		pc_r <= pc_nxt;
		inst1_r <= inst1;
		inst2_r <= inst2;
		inst3_r <= inst3;
	end
end	// always

wire [15:0]	rts;

assign	rts = tmp16 + 16'h0001;

always @(
	pc_mux or rdata or tmp16
	or inst2_r or inst3_r or pc_r or rts
) begin
	case(pc_mux)
	`PCMUX_TMP16:
		new_pc = tmp16;
	`PCMUX_ABS:
		new_pc = {inst3_r, inst2_r};
	`PCMUX_REL:
		new_pc = pc_r + {{8{inst2_r[7]}}, inst2_r};
	`PCMUX_RTS:
		new_pc = rts;
	default:
		new_pc = 16'h0000;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		cur_pc <= 16'h0000;
	else
		cur_pc <= cur_pc_nxt;
end	// always

always @(cur_pc or cpc_mux or pc_r or pc_nxt) begin
	case(cpc_mux)
	`CPCMUX_PCR:
		cur_pc_nxt = pc_r;
	`CPCMUX_PCN:
		cur_pc_nxt = pc_nxt;
	default:
		cur_pc_nxt = cur_pc;
	endcase
end	// always comb

// accumulator
always @(posedge clk or negedge rst) begin
	if(~rst)
		acc <= 8'h00;
	else
		acc <= acc_nxt;
end

always @(acc or acc_mux or outa) begin
	case(acc_mux)
	`ACCMUX_OUTA:
		acc_nxt = outa;
	default:
		acc_nxt = acc;
	endcase
end	// always comb

// index register X
always @(posedge clk or negedge rst) begin
	if(~rst)
		ix <= 8'h00;
	else
		ix <= ix_nxt;
end

always @(ix or ix_mux or outa) begin
	case(ix_mux)
	`IXMUX_OUTA:
		ix_nxt = outa;
	default:
		ix_nxt = ix;
	endcase
end	// always comb

// index register Y
always @(posedge clk or negedge rst) begin
	if(~rst)
		iy <= 8'h00;
	else
		iy <= iy_nxt;
end

always @(iy or iy_mux or outa) begin
	case(iy_mux)
	`IYMUX_OUTA:
		iy_nxt = outa;
	default:
		iy_nxt = iy;
	endcase
end	// always comb

// sp register
always @(posedge clk or negedge rst) begin
	if(~rst)
		sp <= 8'h00;	// ??? post decrement ???
	else
		sp <= sp_nxt;
end

always @(sp or sp_mux or outa) begin
	case(sp_mux)
	`SPMUX_OUTA:
		sp_nxt = outa;
	`SPMUX_INC:
		sp_nxt = sp + 8'h01;
	`SPMUX_DEC:
		sp_nxt = sp - 8'h01;
	default:
		sp_nxt = sp;
	endcase
end	// always comb

// PSR
always @(posedge clk or negedge rst) begin
	if(~rst)
		psr <= 8'h34;	// bit5 = 1, B = 1, I = 1
	else
		psr <= psr_nxt;
end	// always comb

always @(
	psr or upd_flg
	or new_cy or new_ov or new_neg or new_zero
	or psr_mux or outa or rdata
) begin
	psr_nxt = psr;

	if(upd_flg[0])	// update cy
		psr_nxt[0] = new_cy;
	if(upd_flg[1])	// update ov
		psr_nxt[6] = new_ov;
	if(upd_flg[2])	// update neg
		psr_nxt[7] = new_neg;
	if(upd_flg[3])	// update zero
		psr_nxt[1] = new_zero;

	case(psr_mux)
	`PSRMUX_CLI:
		psr_nxt[2] = 1'b0;
	`PSRMUX_SEI:
		psr_nxt[2] = 1'b1;
	`PSRMUX_CLD:
		psr_nxt[3] = 1'b0;
	`PSRMUX_SED:
		psr_nxt[3] = 1'b1;
	`PSRMUX_OUTA:
		psr_nxt = outa;
	`PSRMUX_RDAT:
		psr_nxt = rdata;
	`PSRMUX_BRK: begin
		psr_nxt[2] = 1'b1;	// IRQ mask
		psr_nxt[3] = 1'b0;	// DEC/BIN mode
		psr_nxt[4] = 1'b1;	// BRK
	end
	`PSRMUX_IRQ: begin
		psr_nxt[2] = 1'b1;	// IRQ mask
		psr_nxt[3] = 1'b0;	// DEC/BIN mode
		psr_nxt[4] = 1'b0;	// BRK
	end
	endcase
end	// always comb

assign	cy_bit = psr[0];
assign	imsk_bit = psr[2];
assign	dec_bit = psr[3];

always @(cond or psr) begin
	case(cond)
	`COND_EQ:
		ctrue = psr[1];
	`COND_NEQ:
		ctrue = ~psr[1];
	`COND_CS:
		ctrue = psr[0];
	`COND_CC:
		ctrue = ~psr[0];
	`COND_MI:
		ctrue = psr[7];
	`COND_PL:
		ctrue = ~psr[7];
	`COND_VS:
		ctrue = psr[6];
	`COND_VC:
		ctrue = ~psr[6];
	endcase
end	// always comb

// temporary register
always @(posedge clk or negedge rst) begin
	if(~rst)
		tmp <= 8'h00;
	else
		tmp <= tmp_nxt;
end

always @(tmp or tmp_mux or rdata) begin
	case(tmp_mux)
	`TMUX_RDAT:
		tmp_nxt = rdata;
	default:
		tmp_nxt = tmp;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tmp16 <= 8'h00;
	else
		tmp16 <= tmp16_nxt;
end

always @(tmp16 or tmp16_mux or rdata) begin
	case(tmp16_mux)
	`T16MUX_RDATL:
		tmp16_nxt = {tmp16[15:8], rdata};
	`T16MUX_RDATH:
		tmp16_nxt = {rdata, tmp16[7:0]};
	default:
		tmp16_nxt = tmp16;
	endcase
end	// always comb

always @(
	ina_mux or acc or inst2_r or rdata
	or ix or iy or sp
) begin
	case(ina_mux)
	`AMUX_ACC:
		ina = acc;
	`AMUX_INST2:
		ina = inst2_r;
	`AMUX_RDAT:
		ina = rdata;
	`AMUX_IX:
		ina = ix;
	`AMUX_IY:
		ina = iy;
	`AMUX_SP:
		ina = sp;
	default:
		ina = 8'h00;
	endcase
end	// always comb

always @(
	inb_mux or inst2_r or rdata
) begin
	case(inb_mux)
	`BMUX_INST2:
		inb = inst2_r;
	`BMUX_RDAT:
		inb = rdata;
	`BMUX_ONE:
		inb = 8'h01;
	default:
		inb = 8'h00;
	endcase
end	// always comb

always @(bus_mux) begin
	en = 1'b0;
	wen = 1'b0;

	case(bus_mux)
	`BSMUX_RD: begin
		en = 1'b1;
	end
	`BSMUX_WR: begin
		en = 1'b1;
		wen = 1'b1;
	end
	endcase
end	// always comb


reg [15:0]	ea_ina;
reg [7:0]	ea_inb;
wire [15:0]	effaddr;

always @(
	ea_mux or inst2_r or inst3_r or ix
	or tmp16 or iy or sp or effaddr_r
) begin
	case(ea_mux)
	`EAMUX_ZPX: begin
		ea_ina = {8'h00, inst2_r};
		ea_inb = ix;
	end
	`EAMUX_ABSX: begin
		ea_ina = {inst3_r, inst2_r};
		ea_inb = ix;
	end
	`EAMUX_ZP_1: begin
		ea_ina = {8'h00, inst2_r};
		ea_inb = 8'h01;
	end
	`EAMUX_TMP16Y: begin
		ea_ina = tmp16;
		ea_inb = iy;
	end
	`EAMUX_ABSY: begin
		ea_ina = {inst3_r, inst2_r};
		ea_inb = iy;
	end
	`EAMUX_ABS_1: begin
		ea_ina = {inst3_r, inst2_r};
		ea_inb = 8'h01;
	end
	`EAMUX_SP_1: begin
		ea_ina = {8'h00, sp};
		ea_inb = 8'h01;
	end
	`EAMUX_INCZP: begin
		ea_ina = effaddr_r;
		ea_inb = 8'h01;
	end
	default: begin
		ea_ina = 16'h0000;
		ea_inb = 8'h00;
	end
	endcase
end	// always comb

add_16_8 ea_adder0(
	.ina(ea_ina),
	.inb(ea_inb),
	.out(effaddr)
);

reg [15:0]	effaddr_r;

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		effaddr_r <= 16'h0000;
	end
	else if(ea_sav) begin
		effaddr_r <= effaddr;
	end
end

always @(
	ba_mux or inst2_r or inst3_r
	or ix or iy or tmp16
	or sp or effaddr or effaddr_r
) begin
	case(ba_mux)
	`BAMUX_RSTL:
		addr = 16'hfffc;
	`BAMUX_RSTH:
		addr = 16'hfffd;
	`BAMUX_ABS:
		addr = {inst3_r, inst2_r};
	`BAMUX_ZP:
		addr = {8'h00, inst2_r};
	`BAMUX_ZPEA:
		addr = {8'h00, effaddr[7:0]};
	`BAMUX_TMP16:
		addr = tmp16;
	`BAMUX_EA:
		addr = effaddr;
	`BAMUX_SP:
		addr = {8'h01, sp};
	`BAMUX_SPEA:
		addr = {8'h01, effaddr[7:0]};
	`BAMUX_IRQL:
		addr = 16'hfffe;
	`BAMUX_IRQH:
		addr = 16'hffff;
	`BAMUX_NMIL:
		addr = 16'hfffa;
	`BAMUX_NMIH:
		addr = 16'hfffb;
	default:
		addr = 16'h0000;
	endcase
end	// always comb

wire [15:0]	jsr_addr;

assign	jsr_addr = pc_r - 16'h0001;

wire [15:0]	brk_addr;

assign	brk_addr = pc_r + 16'h0001;

always @(
	wd_mux or acc or ix or iy or psr or outa
	or jsr_addr or brk_addr or cur_pc
) begin
	case(wd_mux)
	`WDMUX_ACC:
		wdata = acc;
	`WDMUX_IX:
		wdata = ix;
	`WDMUX_IY:
		wdata = iy;
	`WDMUX_PSR:
		wdata = psr;
	`WDMUX_OUTA:
		wdata = outa;
	`WDMUX_JSRH:
		wdata = jsr_addr[15:8];
	`WDMUX_JSRL:
		wdata = jsr_addr[7:0];
	`WDMUX_BRKH:
		wdata = brk_addr[15:8];
	`WDMUX_BRKL:
		wdata = brk_addr[7:0];
	`WDMUX_CPCH:
		wdata = cur_pc[15:8];
	`WDMUX_CPCL:
		wdata = cur_pc[7:0];
	default:
		wdata = 8'h00;
	endcase
end	// always comb

endmodule


// End of ac6502_exe_data.v
