//
// ac6502_addsub.v
//
// ac6502 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

module ac6502_addsub(
	op,
	ina,
	inb,
	cy_bit,
	out,
	new_cy,
	new_ov,
	new_neg,
	new_zero
);

input [2:0]	op;
input [7:0]	ina;
input [7:0]	inb;
input	cy_bit;
output [7:0]	out;
output	new_cy;
output	new_ov;
output	new_neg;
output	new_zero;

reg [7:0]	out;
reg	new_cy;
reg	new_ov;
reg	new_neg;
reg	new_zero;

reg [3:0]	inal;
reg [3:0]	inbl;
reg	cy_inl;
reg [3:0]	inah;
reg [3:0]	inbh;
reg	cy_inh;
wire [3:0]	outl;
wire	cy_outl;
wire [3:0]	outh;
wire	cy_outh;

reg [3:0]	da_inl;
reg	da_cyinl;
reg [3:0]	da_inh;
reg	da_cyinh;
wire [3:0]	da_outl;
wire	da_cyoutl;
wire [3:0]	da_outh;
wire	da_cyouth;

// op :
// 000 - ADD bin, 001 - SUB bin,
// 010 - ADC bin, 011 - SBC bin
// 100 - ADC dec, 101 - SBC dec
always @(
	op or ina or inb or cy_bit
	or outl or outh or cy_outl or cy_outh
	or da_cyoutl
) begin
	case(op)
	3'b000: begin	// ADD bin
		inal = ina[3:0];
		inbl = inb[3:0];
		cy_inl = 1'b0;
		da_inl = 4'h0;
		da_cyinl = 1'b0;
		inah = ina[7:4];
		inbh = inb[7:4];
		cy_inh = cy_outl;
		da_inh = 4'h0;
		da_cyinh = 1'b0;
	end
	3'b001: begin	// SUB bin
		inal = ina[3:0];
		inbl = ~inb[3:0];
		cy_inl = 1'b1;
		da_inl = (~4'h0);
		da_cyinl = 1'b1;
		inah = ina[7:4];
		inbh = ~inb[7:4];
		cy_inh = cy_outl;
		da_inh = (~4'h0);
		da_cyinh = 1'b1;
	end
	3'b010: begin	// ADC bin
		inal = ina[3:0];
		inbl = inb[3:0];
		cy_inl = cy_bit;
		da_inl = 4'h0;
		da_cyinl = 1'b0;
		inah = ina[7:4];
		inbh = inb[7:4];
		cy_inh = cy_outl;
		da_inh = 4'h0;
		da_cyinh = 1'b0;
	end
	3'b011: begin	// SBC bin
		inal = ina[3:0];
		inbl = ~inb[3:0];
		cy_inl = cy_bit;	// ???
		da_inl = (~4'h0);
		da_cyinl = 1'b1;
		inah = ina[7:4];
		inbh = ~inb[7:4];
		cy_inh = cy_outl;
		da_inh = (~4'h0);
		da_cyinh = 1'b1;
	end
	3'b100: begin	// ADC dec
		inal = ina[3:0];
		inbl = inb[3:0];
		cy_inl = cy_bit;
		da_inl = (outl > 4'h9 | cy_outl) ? 4'h6 : 4'h0;
		da_cyinl = 1'b0;
		inah = ina[7:4];
		inbh = inb[7:4];
		cy_inh = cy_outl | da_cyoutl;
		da_inh = (outh > 4'h9 | cy_outh) ? 4'h6 : 4'h0;
		da_cyinh = 1'b0;
	end
	3'b101: begin	// SBC dec
		inal = ina[3:0];
		inbl = ~inb[3:0];
		cy_inl = cy_bit;	// ???
		da_inl = (outl > 4'h9 | ~cy_outl) ? (~4'h6) : (~4'h0);
		da_cyinl = 1'b1;
		inah = ina[7:4];
		inbh = ~inb[7:4];
		cy_inh = cy_outl & da_cyoutl;
		da_inh = (outh > 4'h9 | ~cy_outh) ? (~4'h6) : (~4'h0);
		da_cyinh = 1'b1;
	end
	default: begin
		inal = 4'h0;
		inbl = 4'h0;
		cy_inl = 1'b0;
		da_inl = 4'h0;
		da_cyinl = 1'b0;
		inah = 4'h0;
		inbh = 4'h0;
		cy_inh = 1'b0;
		da_inh = 4'h0;
		da_cyinh = 1'b0;
	end
	endcase
end	// comb

// lower 4bit
adder4 add4l(
	.ina(inal),
	.inb(inbl),
	.cy_in(cy_inl),
	.out(outl),
	.cy_out(cy_outl)
);

// higher 4bit
adder4 add4h(
	.ina(inah),
	.inb(inbh),
	.cy_in(cy_inh),
	.out(outh),
	.cy_out(cy_outh)
);

// lower 4bit DAA,DAS
adder4 da4l(
	.ina(outl),
	.inb(da_inl),
	.cy_in(da_cyinl),
	.out(da_outl),
	.cy_out(da_cyoutl)
);

// higher 4bit DAA,DAS
adder4 da4h(
	.ina(outh),
	.inb(da_inh),
	.cy_in(da_cyinh),
	.out(da_outh),
	.cy_out(da_cyouth)
);

always @(da_outh or da_outl) begin
	out = {da_outh, da_outl};
end	// always comb

always @(op or cy_outh or da_cyouth) begin
	if(op[0])	// sub
		new_cy = cy_outh & da_cyouth;
	else		// add
		new_cy = cy_outh | da_cyouth;
end	// always comb

always @(inah or inbh or out) begin
	new_ov = (inah[3] & inbh[3] & ~out[7])
			| (~inah[3] & ~inbh[3] & out[7]);
end	// always comb

always @(out) begin
	new_neg = out[7];
	new_zero = ~(| out);
end	// always comb

endmodule

// End of ac6502_addsub.v
