//
// ac51_tmr01.v
//
// ac51 microcontroller core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

module ac51_tmr01(
	clk,
	rst,
	ioaddr,
	iowdata,
	iordata,
	iowen,
	sck,
	eint_i,
	eint_req,
	eint_ack,
	tc_i,
	tf_req,
	tf_ack
);

input	clk;
input	rst;
input [6:0]	ioaddr;
input [7:0]	iowdata;
output [7:0]	iordata;
input	iowen;
output	sck;
input [1:0]	eint_i;
output [1:0]	eint_req;
input [1:0]	eint_ack;
input [1:0]	tc_i;
output [1:0]	tf_req;
input [1:0]	tf_ack;

reg [6:0]	ioaddr_d;
wire	tcon_enb;
wire	tmod_enb;
wire	tl0_enb;
wire	th0_enb;
wire	tl1_enb;
wire	th1_enb;

wire	tcon_wen;
wire	tmod_wen;
wire	tl0_wen;
wire	th0_wen;
wire	tl1_wen;
wire	th1_wen;

// TCON bits
wire [7:0]	tcon;
reg	it0;
wire	ie0;
reg	it1;
wire	ie1;
reg	tr0;
reg	tf0;
reg	tr1;
reg	tf1;

// TMOD bits
wire [7:0]	tmod;
reg	gt1;
reg	c_t1;
reg [1:0]	mode1;
reg	gt0;
reg	c_t0;
reg [1:0]	mode0;

reg [7:0]	tl0;
reg [7:0]	th0;
reg [7:0]	tl1;
reg [7:0]	th1;

reg [1:0]	eint0_seq;
reg [1:0]	eint1_seq;
wire	eint0_fall;
wire	eint1_fall;
reg	eint0_req;
reg	eint1_req;

reg [1:0]	tc0_seq;
reg [1:0]	tc1_seq;
wire	tc0_fall;
wire	tc1_fall;
reg [3:0]	t0_presc;
reg [3:0]	t1_presc;
reg	t0_run;
reg	t0_cnt;
reg	tl0_cnt;
reg	th0_cnt;
reg	tf0_set;
reg	t1_run;
reg	t1_cnt;
reg	tl1_cnt;
reg	th1_cnt;
reg	tf1_set;

reg	sck;


always @(posedge clk or negedge rst) begin
	if(~rst)
		ioaddr_d <= 7'h00;
	else
		ioaddr_d <= ioaddr;
end	// always

assign	tcon_enb = (ioaddr_d == 7'h08);	// TCON @8'h88
assign	tmod_enb = (ioaddr_d == 7'h09);	// TMOD @8'h89
assign	tl0_enb = (ioaddr_d == 7'h0a);	// TL0 @8'h8a
assign	th0_enb = (ioaddr_d == 7'h0c);	// TH0 @8'h8c
assign	tl1_enb = (ioaddr_d == 7'h0b);	// TL1 @8'h8b
assign	th1_enb = (ioaddr_d == 7'h0d);	// TH1 @8'h8d

assign	tcon = {tf1, tr1, tf0, tr0, ie1, it1, ie0, it0};
assign	tmod = {gt1, c_t1, mode1, gt0, c_t0, mode0};

assign	iordata = ({8{tcon_enb}} & tcon)
					| ({8{tmod_enb}} & tmod)
					| ({8{tl0_enb}} & tl0)
					| ({8{th0_enb}} & th0)
					| ({8{tl1_enb}} & tl1)
					| ({8{th1_enb}} & th1);

assign	tcon_wen = ((ioaddr == 7'h08) & iowen);	// TCON @8'h88
assign	tmod_wen = ((ioaddr == 7'h09) & iowen);	// TMOD @8'h89
assign	tl0_wen = ((ioaddr == 7'h0a) & iowen);	// TL0 @8'h8a
assign	th0_wen = ((ioaddr == 7'h0c) & iowen);	// TH0 @8'h8c
assign	tl1_wen = ((ioaddr == 7'h0b) & iowen);	// TL1 @8'h8b
assign	th1_wen = ((ioaddr == 7'h0d) & iowen);	// TH1 @8'h8d

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		tr1 <= 1'b0;
		tr0 <= 1'b0;
		it1 <= 1'b0;
		it0 <= 1'b0;
	end
	else if(tcon_wen) begin
		tr1 <= iowdata[6];
		tr0 <= iowdata[4];
		it1 <= iowdata[2];
		it0 <= iowdata[0];
	end
end	// always

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		gt1 <= 1'b0;
		c_t1 <= 1'b0;
		mode1 <= 2'b00;
		gt0 <= 1'b0;
		c_t0 <= 1'b0;
		mode0 <= 2'b00;
	end
	else if(tmod_wen) begin
		gt1 <= iowdata[7];
		c_t1 <= iowdata[6];
		mode1 <= iowdata[5:4];
		gt0 <= iowdata[3];
		c_t0 <= iowdata[2];
		mode0 <= iowdata[1:0];
	end
end	// always

assign	ie0 = it0 ? eint0_req : eint_i[0];
assign	ie1 = it1 ? eint1_req : eint_i[1];

// sample ext int pin status
always @(posedge clk or negedge rst) begin
	if(~rst) begin
		eint0_seq <= 2'b11;
		eint1_seq <= 2'b11;
	end
	else begin
		eint0_seq <= {eint0_seq[0], eint_i[0]};
		eint1_seq <= {eint1_seq[0], eint_i[1]};
	end
end	// always

assign	eint0_fall = eint0_seq[1] & ~eint0_seq[0];
assign	eint1_fall = eint1_seq[1] & ~eint1_seq[0];

always @(posedge clk or negedge rst) begin
	if(~rst)
		eint0_req <= 1'b0;
	else if(eint_ack[0])
		eint0_req <= 1'b0;
	else if(eint0_fall)
		eint0_req <= 1'b1;
end	// always

assign	eint_req[0] = eint0_req;

always @(posedge clk or negedge rst) begin
	if(~rst)
		eint1_req <= 1'b0;
	else if(eint_ack[1])
		eint1_req <= 1'b0;
	else if(eint1_fall)
		eint1_req <= 1'b1;
end	// always

assign	eint_req[1] = eint1_req;

// external clk
always @(posedge clk or negedge rst) begin
	if(~rst) begin
		tc0_seq <= 2'b11;
		tc1_seq <= 2'b11;
	end
	else begin
		tc0_seq <= {tc0_seq[0], tc_i[0]};
		tc1_seq <= {tc1_seq[0], tc_i[1]};
	end
end	// always

assign	tc0_fall = tc0_seq[1] & ~tc0_seq[0];
assign	tc1_fall = tc1_seq[1] & ~tc1_seq[0];

// internal clk
always @(posedge clk or negedge rst) begin
	if(~rst)
		t0_presc <= 4'd11;
	else if(t0_presc == 4'h0)
		t0_presc <= 4'd11;
	else
		t0_presc <= t0_presc - 4'h1;
end	// always

always @(posedge clk or negedge rst) begin
	if(~rst)
		t1_presc <= 4'd11;
	else if(t1_presc == 4'h0)
		t1_presc <= 4'd11;
	else
		t1_presc <= t1_presc - 4'h1;
end	// always

// Timer0

always @(tr0 or gt0 or eint_i) begin
	t0_run = tr0;

	if(gt0) begin
		if(~eint_i[0]) t0_run = 1'b0;
	end
end	// always comb

always @(t0_run or c_t0 or t0_presc or tc0_fall) begin
	t0_cnt = 1'b0;

	if(t0_run) begin
		if(c_t0) begin
			t0_cnt = tc0_fall;
		end
		else begin
			if(t0_presc == 4'h0) t0_cnt = 1'b1;
		end
	end
end	// always comb

always @(t0_cnt) begin
	tl0_cnt = t0_cnt;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tl0 <= 8'h00;
	else if(tl0_wen)
		tl0 <= iowdata;
	else if(tl0_cnt) begin
		if((mode0 == 2'b10) & (tl0 == 8'hff))
			tl0 <= th0;
		else
			tl0 <= tl0 + 8'h01;
	end
end	// always

always @(
	mode0 or tl0 or tl0_cnt
	or tr1 or t0_presc
) begin
	case(mode0)
	2'b00,	// Mode0 13bit
	2'b01:	// Mode1 16bit
		th0_cnt = ((tl0 == 8'hff) & tl0_cnt);
	2'b10:	// Mode2 8bit with auto-reload
		th0_cnt = 1'b0;
	2'b11:	// Mode3 8bit+8bit
		th0_cnt = tr1 & (t0_presc == 4'h0);
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		th0 <= 8'h00;
	else if(th0_wen)
		th0 <= iowdata;
	else if(th0_cnt)
		th0 <= th0 + 8'h01;
end	// always

always @(mode0 or th0 or th0_cnt or tl0 or tl0_cnt) begin
	case(mode0)
	2'b00:	// Mode0 13bit
		tf0_set = ((th0[4:0] == 5'h1f) & th0_cnt);
	2'b01:	// Mode1 16bit
		tf0_set = ((th0 == 8'hff) & th0_cnt);
	2'b10,	// Mode2 8bit with auto-reload
	2'b11:	// Mode3 8bit+8bit
		tf0_set = ((tl0 == 8'hff) & tl0_cnt);
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tf0 <= 1'b0;
	else if(tf_ack[0])
		tf0 <= 1'b0;
	else if(tf0_set)
		tf0 <= 1'b1;
end	// always

assign	tf_req[0] = tf0;

// Timer 1

always @(tr1 or gt1 or eint_i) begin
	t1_run = tr1;

	if(gt1) begin
		if(~eint_i[1]) t1_run = 1'b0;
	end
end	// always comb

always @(t1_run or c_t1 or t1_presc or tc1_fall) begin
	t1_cnt = 1'b0;

	if(t1_run) begin
		if(c_t1) begin
			t1_cnt = tc1_fall;
		end
		else begin
			if(t1_presc == 4'h0) t1_cnt = 1'b1;
		end
	end
end	// always comb

always @(t1_cnt or mode1 or mode0 or t1_presc) begin
	if(mode1 == 2'b11) begin	// Mode3 stop
		tl1_cnt = 1'b0;
	end
	else begin
		if(mode0 == 2'b11)	// Mode3 8bit+8bit
			tl1_cnt = (t1_presc == 4'h0);
		else
			tl1_cnt = t1_cnt;
	end
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tl1 <= 8'h00;
	else if(tl1_wen)
		tl1 <= iowdata;
	else if(tl1_cnt) begin
		if((mode1 == 2'b10) & (tl1 == 8'hff))
			tl1 <= th1;
		else
			tl1 <= tl1 + 8'h01;
	end
end	// always

always @(mode1 or tl1 or tl1_cnt) begin
	case(mode1)
	2'b00,	// Mode0 13bit
	2'b01:	// Mode1 16bit
		th1_cnt = ((tl1 == 8'hff) & tl1_cnt);
	2'b10,	// Mode2 8bit with auto-reload
	2'b11:	// Mode3 stop
		th1_cnt = 1'b0;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		th1 <= 8'h00;
	else if(th1_wen)
		th1 <= iowdata;
	else if(th1_cnt)
		th1 <= th1 + 8'h01;
end	// always

always @(
	mode1 or mode0
	or th1 or th1_cnt or tl1 or tl1_cnt
	or th0 or th0_cnt
) begin
	if(mode0 == 2'b11) begin	// Mode3 8bit+8bit
		tf1_set = ((th0 == 8'hff) & th0_cnt);
	end
	else begin
		case(mode1)
		2'b00:	// Mode0 13bit
			tf1_set = ((th1[4:0] == 5'h1f) & th1_cnt);
		2'b01:	// Mode1 16bit
			tf1_set = ((th1 == 8'hff) & th1_cnt);
		2'b10:	// Mode2 8bit with auto-reload
			tf1_set = ((tl1 == 8'hff) & tl1_cnt);
		2'b11:	// Mode3 stop
			tf1_set = 1'b0;
		endcase
	end
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tf1 <= 1'b0;
	else if(tf_ack[1])
		tf1 <= 1'b0;
	else if(tf1_set)
		tf1 <= 1'b1;
end	// always

assign	tf_req[1] = tf1;

always @(
	mode1 or th1 or th1_cnt or tl1 or tl1_cnt
) begin
	case(mode1)
	2'b00:	// Mode0 13bit
		sck = ((th1[4:0] == 5'h1f) & th1_cnt);
	2'b01:	// Mode1 16bit
		sck = ((th1 == 8'hff) & th1_cnt);
	2'b10:	// Mode2 8bit with auto-reload
		sck = ((tl1 == 8'hff) & tl1_cnt);
	2'b11:	// Mode3 stop
		sck = 1'b0;
	endcase
end	// always comb

endmodule


// End of ac51_tmr01.v
