//
// ac51_exe_data.v
//
// ac51 microcontroller core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac51_defs.v"

module ac51_exe_data(
	clk,
	rst,
	pc_nxt,
	inst1,
	inst2,
	inst3,
	inst1_r,
	inst_ack,
	pc_mux,
	cpc_mux,
	sp_mux,
	dptr_mux,
	new_pc,
	acc_mux,
	acc_zero,
	breg_mux,
	ina_mux,
	inb_mux,
	ina,
	inb,
	div_tmp_i,
	cy_bit,
	ac_bit,
	ov_bit,
	outa,
	outb,
	div_tmp_o,
	new_cy,
	new_ac,
	new_ov,
	tmp_mux,
	tmp_zero,
	bit_data,
	flg_upd,
	ea_mux,
	ibus_mux,
	iwd_mux,
	iaddr,
	iwdata,
	irdata,
	iwen,
	ioaddr,
	iowdata,
	iordata,
	iormw,
	iowen,
	pbus_mux,
	paddr,
	pen,
	prdata,
	xbus_mux,
	xaddr,
	p2,
	xen,
	xwdata,
	xrdata,
	xwen,
	int_pri,
	int_enb
);

input	clk;
input	rst;
input [15:0]	pc_nxt;
input [7:0]	inst1;
input [7:0]	inst2;
input [7:0]	inst3;
output [7:0]	inst1_r;
input	inst_ack;
input [`PCMUX_LEN - 1:0]	pc_mux;
input [`CPCMUX_LEN - 1:0]	cpc_mux;
input [`SPMUX_LEN - 1:0]	sp_mux;
input [`DPMUX_LEN - 1:0]	dptr_mux;
output [15:0]	new_pc;
input [`ACCMUX_LEN - 1:0]	acc_mux;
output	acc_zero;
input [`BRGMUX_LEN - 1:0]	breg_mux;
input [`AMUX_LEN - 1:0]	ina_mux;
input [`BMUX_LEN - 1:0]	inb_mux;
output [7:0]	ina;
output [7:0]	inb;
output [15:0]	div_tmp_i;
output	cy_bit;
output	ac_bit;
output	ov_bit;
input [7:0]	outa;
input [7:0]	outb;
input [15:0]	div_tmp_o;
input	new_cy;
input	new_ac;
input	new_ov;
input [`TMUX_LEN - 1:0]	tmp_mux;
output	tmp_zero;
output	bit_data;
input [2:0]	flg_upd;	// bit0 - ac, bit1 - cy, bit2 - ov
input [`EAMUX_LEN - 1:0]	ea_mux;
input [`IMUX_LEN - 1:0]	ibus_mux;
input [`IWDMUX_LEN - 1:0]	iwd_mux;
output [7:0]	iaddr;
output [7:0]	iwdata;
input [7:0]	irdata;
output	iwen;
output [6:0]	ioaddr;
output [7:0]	iowdata;
input [7:0]	iordata;
output	iormw;
output	iowen;
input [`PMUX_LEN - 1:0]	pbus_mux;
output [15:0]	paddr;
output	pen;
input [7:0]	prdata;
input [`XMUX_LEN - 1:0]	xbus_mux;
output [15:0]	xaddr;
input [7:0]	p2;	// port 2 for Ri-indirect xaddr
output	xen;
output [7:0]	xwdata;
input [7:0]	xrdata;
output	xwen;
output [5:0]	int_pri;
output [5:0]	int_enb;


reg [15:0]	pc_r;
reg [7:0]	inst1_r;
reg [7:0]	inst2_r;
reg [7:0]	inst3_r;

reg [15:0]	new_pc;

reg [15:0]	cur_pc;
reg [15:0]	cur_pc_nxt;

reg [15:0]	ea_base;
reg [15:0]	ea_indx;
wire [15:0]	effaddr;

// internal data area interface
reg [7:0]	iaddr;
reg	idirect;
reg	irmw;
reg [7:0]	iwdata;
reg	iwen;

// sfr area interface
wire [6:0]	ioaddr;
wire [7:0]	iowdata;
wire [7:0]	iordata;
wire	iormw;
reg	iowen;

reg	iwen_i;

// program area interface
reg [15:0]	paddr;
reg	pen;

// ex data area interface
reg [15:0]	xaddr;
reg	xen;
reg [7:0]	xwdata;
reg	xwen;

reg [7:0]	bnk0r0;
reg [7:0]	bnk0r1;
reg [7:0]	bnk1r0;
reg [7:0]	bnk1r1;
reg [7:0]	bnk2r0;
reg [7:0]	bnk2r1;
reg [7:0]	bnk3r0;
reg [7:0]	bnk3r1;
reg [7:0]	bnk0r0_nxt;
reg [7:0]	bnk0r1_nxt;
reg [7:0]	bnk1r0_nxt;
reg [7:0]	bnk1r1_nxt;
reg [7:0]	bnk2r0_nxt;
reg [7:0]	bnk2r1_nxt;
reg [7:0]	bnk3r0_nxt;
reg [7:0]	bnk3r1_nxt;

reg [7:0]	acc;
reg [7:0]	acc_nxt;
reg [7:0]	acc_nxt2;

reg [7:0]	b_reg;
reg [7:0]	b_nxt;
reg [7:0]	b_nxt2;

reg [7:0]	sp;
reg [7:0]	sp_nxt;
reg [7:0]	sp_nxt2;

reg [15:0]	dptr;
reg [15:0]	dptr_nxt;
reg [15:0]	dptr_nxt2;

reg	cy_bit;
reg	ac_bit;
reg	f0_bit;
reg [1:0]	rs_bits;
reg	ov_bit;
reg	p_bit;
wire [7:0]	psw;
reg	cy_nxt;
reg	ac_nxt;
reg	f0_nxt;
reg [1:0]	rs_nxt;
reg	ov_nxt;
reg [7:0]	psw_nxt2;

reg [6:0]	ie_reg;
reg [5:0]	ip_reg;
reg [6:0]	ie_nxt;
reg [5:0]	ip_nxt;
reg [6:0]	ie_nxt2;
reg [5:0]	ip_nxt2;

// ALU interface
reg [7:0]	ina;
reg [7:0]	inb;

// temporary reg
reg [7:0]	tmp;
reg [7:0]	tmp_nxt;

reg [15:0]	div_tmp;

reg	bit_data;

reg [7:0]	rn_addr;
reg [7:0]	ri_addr;
reg [7:0]	bit_addr;

reg [7:0]	iaddr_d;
reg	idirect_d;
reg [7:0]	irdata_i;


always @(posedge clk or negedge rst) begin
	if(~rst) begin
		pc_r <= 16'h0000;
		inst1_r <= 8'h00;
		inst2_r <= 8'h00;
		inst3_r <= 8'h00;
	end
	else if(inst_ack) begin
		pc_r <= pc_nxt;
		inst1_r <= inst1;
		inst2_r <= inst2;
		inst3_r <= inst3;
	end
end	// always

always @(posedge clk or negedge rst) begin
	if(~rst)
		ac_bit <= 1'b0;
	else
		ac_bit <= ac_nxt;
end	// always

always @(psw_nxt2 or flg_upd or new_ac) begin
	ac_nxt = psw_nxt2[6];

	if(flg_upd[0])
		ac_nxt = new_ac;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		cy_bit <= 1'b0;
	else
		cy_bit <= cy_nxt;
end	// always

always @(psw_nxt2 or flg_upd or new_cy) begin
	cy_nxt = psw_nxt2[7];

	if(flg_upd[1])
		cy_nxt = new_cy;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		ov_bit <= 1'b0;
	else
		ov_bit <= ov_nxt;
end	// always

always @(psw_nxt2 or flg_upd or new_ov) begin
	ov_nxt = psw_nxt2[2];

	if(flg_upd[2])
		ov_nxt = new_ov;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		f0_bit <= 1'b0;
	else
		f0_bit <= f0_nxt;
end	// always

always @(psw_nxt2) begin
	f0_nxt = psw_nxt2[5];
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		rs_bits <= 2'b00;
	else
		rs_bits <= rs_nxt;
end	// always

always @(psw_nxt2) begin
	rs_nxt = psw_nxt2[4:3];
end	// always comb

always @(acc) begin
	p_bit = ^acc;
end	// always comb

assign	psw = {cy_bit, ac_bit, f0_bit, rs_bits, ov_bit, 1'b0, p_bit};

always @(posedge clk or negedge rst) begin
	if(~rst)
		ie_reg <= 7'b0000000;
	else
		ie_reg <= ie_nxt;
end	// always

assign	int_enb = {6{ie_reg[6]}} & ie_reg[5:0];

always @(ie_nxt2) begin
	ie_nxt = ie_nxt2;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		ip_reg <= 6'b00_0000;
	else
		ip_reg <= ip_nxt;
end	// always

assign	int_pri = ip_reg;

always @(ip_nxt2) begin
	ip_nxt = ip_nxt2;
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		acc <= 1'b0;
	else
		acc <= acc_nxt;
end	// always

assign	acc_zero = ~(| acc);

always @(
	acc_nxt2 or acc_mux or irdata_i or outa or outb
	or inst2_r or prdata or xrdata
) begin
	case(acc_mux)
	`ACCMUX_IDATA:
		acc_nxt = irdata_i;
	`ACCMUX_OUTA:
		acc_nxt = outa;
	`ACCMUX_ZERO:
		acc_nxt = 8'h00;
	`ACCMUX_INST2:
		acc_nxt = inst2_r;
	`ACCMUX_PDATA:
		acc_nxt = prdata;
	`ACCMUX_XDATA:
		acc_nxt = xrdata;
	`ACCMUX_OUTB:
		acc_nxt = outb;
	default:
		acc_nxt = acc_nxt2;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		b_reg <= 8'h00;
	else
		b_reg <= b_nxt;
end	// always

always @(b_nxt2 or breg_mux or outa or outb) begin
	case(breg_mux)
	`BRGMUX_OUTB:
		b_nxt = outb;
	`BRGMUX_OUTA:
		b_nxt = outa;
	default:
		b_nxt = b_nxt2;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		sp <= 8'h07;
	else
		sp <= sp_nxt;
end	// always

always @(sp_nxt2 or sp_mux or outa) begin
	case(sp_mux)
	`SPMUX_OUTA:
		sp_nxt = outa;
	default:
		sp_nxt = sp_nxt2;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		dptr <= 16'h0000;
	else
		dptr <= dptr_nxt;
end	// always

always @(
	dptr_nxt2 or dptr_mux or inst2_r or inst3_r or effaddr
) begin
	case(dptr_mux)
	`DPMUX_INST23:
		dptr_nxt = {inst2_r, inst3_r};
	`DPMUX_EA:
		dptr_nxt = effaddr;
	default:
		dptr_nxt = dptr_nxt2;
	endcase
end	// always comb

always @(
	ea_mux or dptr or pc_r or acc
	or inst2_r or inst3_r
) begin
	case(ea_mux)
	`EAMUX_INCDPTR: begin
		ea_base = dptr;
		ea_indx = 16'h0001;
	end
	`EAMUX_ADPTR: begin
		ea_base = dptr;
		ea_indx = {8'h00, acc};
	end
	`EAMUX_APC: begin
		ea_base = pc_r;
		ea_indx = {8'h00, acc};
	end
	`EAMUX_REL2: begin
		ea_base = pc_r;
		ea_indx = {{8{inst2_r[7]}}, inst2_r};
	end
	`EAMUX_REL3: begin
		ea_base = pc_r;
		ea_indx = {{8{inst3_r[7]}}, inst3_r};
	end
	default: begin
		ea_base = 16'h0000;
		ea_indx = 8'h00;
	end
	endcase
end	// always comb

assign	effaddr = ea_base + ea_indx;

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		bnk0r0 <= 8'h00;
		bnk0r1 <= 8'h00;
		bnk1r0 <= 8'h00;
		bnk1r1 <= 8'h00;
		bnk2r0 <= 8'h00;
		bnk2r1 <= 8'h00;
		bnk3r0 <= 8'h00;
		bnk3r1 <= 8'h00;
	end
	else begin
		bnk0r0 <= bnk0r0_nxt;
		bnk0r1 <= bnk0r1_nxt;
		bnk1r0 <= bnk1r0_nxt;
		bnk1r1 <= bnk1r1_nxt;
		bnk2r0 <= bnk2r0_nxt;
		bnk2r1 <= bnk2r1_nxt;
		bnk3r0 <= bnk3r0_nxt;
		bnk3r1 <= bnk3r1_nxt;
	end
end	// always

always @(
	pc_mux or pc_r or inst1_r or inst2_r or inst3_r
	or tmp or irdata_i or effaddr
) begin
	case(pc_mux)
	`PCMUX_ABS:
		new_pc = {pc_r[15:11], inst1_r[7:5], inst2_r};
	`PCMUX_LONG:
		new_pc = {inst2_r, inst3_r};
	`PCMUX_POP:
		new_pc = {tmp, irdata_i};
	`PCMUX_EA:
		new_pc = effaddr;
	`PCMUX_INT0:
		new_pc = 16'h0003;
	`PCMUX_INT1:
		new_pc = 16'h000b;
	`PCMUX_INT2:
		new_pc = 16'h0013;
	`PCMUX_INT3:
		new_pc = 16'h001b;
	`PCMUX_INT4:
		new_pc = 16'h0023;
	`PCMUX_INT5:
		new_pc = 16'h002b;
	default:
		new_pc = 16'h0000;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		cur_pc <= 16'h0000;
	else
		cur_pc <= cur_pc_nxt;
end	// always

always @(cur_pc or cpc_mux or pc_r or pc_nxt) begin
	case(cpc_mux)
	`CPCMUX_PCR:
		cur_pc_nxt = pc_r;
	`CPCMUX_PCN:
		cur_pc_nxt = pc_nxt;
	default:
		cur_pc_nxt = cur_pc;
	endcase
end	// always comb

always @(ina_mux or acc or irdata_i or sp) begin
	case(ina_mux)
	`AMUX_ACC:
		ina = acc;
	`AMUX_IDATA:
		ina = irdata_i;
	`AMUX_SP:
		ina = sp;
	default:
		ina = 8'h00;
	endcase
end	// always comb

always @(
	inb_mux or inst2_r or inst3_r
	or irdata_i or b_reg or acc
) begin
	case(inb_mux)
	`BMUX_INST2:
		inb = inst2_r;
	`BMUX_IDATA:
		inb = irdata_i;
	`BMUX_ONE:
		inb = 8'h01;
	`BMUX_BRG:
		inb = b_reg;
	`BMUX_ACC:
		inb = acc;
	`BMUX_INST3:
		inb = inst3_r;
	default:
		inb = 8'h00;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tmp <= 8'h00;
	else
		tmp <= tmp_nxt;
end	// always

always @(tmp or tmp_mux or outa or irdata_i) begin
	case(tmp_mux)
	`TMUX_OUTA:
		tmp_nxt = outa;
	`TMUX_IDATA:
		tmp_nxt = irdata_i;
	default:
		tmp_nxt = tmp;
	endcase
end	// always comb

assign	tmp_zero = ~(| tmp);

always @(posedge clk or negedge rst) begin
	if(~rst)
		div_tmp <= 16'h0000;
	else
		div_tmp <= div_tmp_o;
end	// always

assign	div_tmp_i = div_tmp;

always @(rs_bits or inst1_r) begin
	rn_addr = {3'b000, rs_bits, inst1_r[2:0]};
end	// always comb

always @(
	rs_bits or inst1_r
	or bnk0r0 or bnk0r1 or bnk1r0 or bnk1r1
	or bnk2r0 or bnk2r1 or bnk3r0 or bnk3r1
) begin
	ri_addr = 8'h00;

	case({rs_bits, inst1_r[0]})
	3'b000:	ri_addr = bnk0r0;
	3'b001:	ri_addr = bnk0r1;
	3'b010:	ri_addr = bnk1r0;
	3'b011:	ri_addr = bnk1r1;
	3'b100:	ri_addr = bnk2r0;
	3'b101:	ri_addr = bnk2r1;
	3'b110:	ri_addr = bnk3r0;
	3'b111:	ri_addr = bnk3r1;
	endcase
end	// always comb

always @(inst2_r) begin
	bit_addr = 8'h00;

	if(inst2_r[7]) begin	// SFR
		bit_addr = {inst2_r[7:3], 3'b000};
	end
	else begin				// 0x20-0x2f
		bit_addr = {4'b0010, inst2_r[6:3]};
	end
end	// always comb

always @(
	ibus_mux or inst2_r or inst3_r
	or rn_addr or ri_addr or bit_addr
	or sp
	or iwd_mux or acc or outa or outb
	or pc_r or irdata_i or cur_pc
) begin
	iaddr = 8'h00;
	idirect = 1'b0;
	irmw = 1'b0;
	iwdata = 8'h00;
	iwen_i = 1'b0;

	case(ibus_mux)
	`IMUX_DRD2: begin
		iaddr = inst2_r;
		idirect = 1'b1;
	end
	`IMUX_DWR2: begin
		iaddr = inst2_r;
		idirect = 1'b1;
		iwen_i = 1'b1;
	end
	`IMUX_RNRD: begin
		iaddr = rn_addr;
	end
	`IMUX_RIRD:	begin
		iaddr = ri_addr;
	end
	`IMUX_BTRD2: begin
		iaddr = bit_addr;
		idirect = 1'b1;
	end
	`IMUX_BTWR2: begin
		iaddr = bit_addr;
		idirect = 1'b1;
		iwen_i = 1'b1;
	end
	`IMUX_RNWR: begin
		iaddr = rn_addr;
		iwen_i = 1'b1;
	end
	`IMUX_RIWR: begin
		iaddr = ri_addr;
		iwen_i = 1'b1;
	end
	`IMUX_OUTA: begin
		iaddr = outa;
		iwen_i = 1'b1;
	end
	`IMUX_SP: begin
		iaddr = sp;
	end
	`IMUX_DWR3: begin
		iaddr = inst3_r;
		idirect = 1'b1;
		iwen_i = 1'b1;
	end
	`IMUX_DRD2RMW: begin
		iaddr = inst2_r;
		idirect = 1'b1;
		irmw = 1'b1;
	end
	`IMUX_BTRD2RMW: begin
		iaddr = bit_addr;
		idirect = 1'b1;
		irmw = 1'b1;
	end
	endcase

	case(iwd_mux)
	`IWDMUX_INST3:
		iwdata = inst3_r;
	`IWDMUX_ACC:
		iwdata = acc;
	`IWDMUX_OUTA:
		iwdata = outa;
	`IWDMUX_INST2:
		iwdata = inst2_r;
	`IWDMUX_PCL:
		iwdata = pc_r[7:0];
	`IWDMUX_PCH:
		iwdata = pc_r[15:8];
	`IWDMUX_IDATA:
		iwdata = irdata_i;
	`IWDMUX_OUTB:
		iwdata = outb;
	`IWDMUX_CPCL:
		iwdata = cur_pc[7:0];
	`IWDMUX_CPCH:
		iwdata = cur_pc[15:8];
	endcase
end	// always comb

always @(
	bnk0r0 or bnk0r1 or bnk1r0 or bnk1r1
	or bnk2r0 or bnk2r1 or bnk3r0 or bnk3r1
	or iwen_i or idirect or iaddr or iwdata
) begin
	bnk0r0_nxt = bnk0r0;
	bnk0r1_nxt = bnk0r1;
	bnk1r0_nxt = bnk1r0;
	bnk1r1_nxt = bnk1r1;
	bnk2r0_nxt = bnk2r0;
	bnk2r1_nxt = bnk2r1;
	bnk3r0_nxt = bnk3r0;
	bnk3r1_nxt = bnk3r1;
	iwen = iwen_i;

	if(iwen_i) begin
		if({iaddr[7:5], iaddr[2:1]} == 5'b00000) begin
			iwen = 1'b0;
			case({iaddr[4:3], iaddr[0]})
			3'b000:	bnk0r0_nxt = iwdata;
			3'b001:	bnk0r1_nxt = iwdata;
			3'b010:	bnk1r0_nxt = iwdata;
			3'b011:	bnk1r1_nxt = iwdata;
			3'b100:	bnk2r0_nxt = iwdata;
			3'b101:	bnk2r1_nxt = iwdata;
			3'b110:	bnk3r0_nxt = iwdata;
			3'b111:	bnk3r1_nxt = iwdata;
			endcase
		end

		if(idirect & iaddr[7]) begin
			iwen = 1'b0;
		end
	end
end	// always comb

always @(
	sp or dptr or ie_reg or ip_reg or psw or acc or b_reg
	or idirect or iwen_i or iaddr or iwdata
) begin
	// default
	sp_nxt2 = sp;
	dptr_nxt2 = dptr;
	ie_nxt2 = ie_reg;
	ip_nxt2 = ip_reg;
	psw_nxt2 = psw;
	acc_nxt2 = acc;
	b_nxt2 = b_reg;
	iowen = 1'b0;

	if(idirect & iwen_i) begin
		if(iaddr[7]) begin
			case(iaddr[6:0])
			7'h01:			// SP
				sp_nxt2 = iwdata;
			7'h02:			// DPL
				dptr_nxt2 = {dptr[15:8], iwdata};
			7'h03:			// DPH
				dptr_nxt2 = {iwdata, dptr[7:0]};
			7'h28:			// IE
				ie_nxt2 = {iwdata[7], iwdata[5:0]};
			7'h38:			// IP
				ip_nxt2 = iwdata[5:0];
			7'h50: begin	// PSW
				psw_nxt2 = iwdata;
			end
			7'h60:			// ACC
				acc_nxt2 = iwdata;
			7'h70:	// B
				b_nxt2 = iwdata;
			default:
				iowen = 1'b1;
			endcase
		end
	end
end	// always comb

assign	ioaddr = iaddr[6:0];
assign	iowdata = iwdata;
assign	iormw = irmw;

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		iaddr_d <= 8'h00;
		idirect_d <= 1'b0;
	end
	else begin
		iaddr_d <= iaddr;
		idirect_d <= idirect;
	end
end	// always

always @(
	irdata or idirect_d or iaddr_d or iordata
	or bnk0r0 or bnk0r1 or bnk1r0 or bnk1r1
	or bnk2r0 or bnk2r1 or bnk3r0 or bnk3r1
	or sp or dptr or ie_reg or ip_reg
	or psw or acc or b_reg or iordata
) begin
	irdata_i = irdata;

	if({iaddr_d[7:5], iaddr_d[2:1]} == 5'b00000) begin
		case({iaddr_d[4:3], iaddr_d[0]})
		3'b000:	irdata_i = bnk0r0;
		3'b001:	irdata_i = bnk0r1;
		3'b010:	irdata_i = bnk1r0;
		3'b011:	irdata_i = bnk1r1;
		3'b100:	irdata_i = bnk2r0;
		3'b101:	irdata_i = bnk2r1;
		3'b110:	irdata_i = bnk3r0;
		3'b111:	irdata_i = bnk3r1;
		endcase
	end

	if(idirect_d) begin
		if(iaddr_d[7]) begin
			case(iaddr_d[6:0])
			7'h01:	// SP
				irdata_i = sp;
			7'h02:	// DPL
				irdata_i = dptr[7:0];
			7'h03:	// DPH
				irdata_i = dptr[15:8];
			7'h28:	// IE
				irdata_i = {ie_reg[6], 1'b0, ie_reg[5:0]};
			7'h38:	// IP
				irdata_i = {2'b00, ip_reg[5:0]};
			7'h50:	// PSW
				irdata_i = psw;
			7'h60:	// ACC
				irdata_i = acc;
			7'h70:	// B
				irdata_i = b_reg;
			default:
				irdata_i = iordata;
			endcase
		end
	end
end	// always comb

always @(inst2_r or irdata_i) begin
	bit_data = 1'b0;

	case(inst2_r[2:0])
	3'b000:	bit_data = irdata_i[0];
	3'b001:	bit_data = irdata_i[1];
	3'b010:	bit_data = irdata_i[2];
	3'b011:	bit_data = irdata_i[3];
	3'b100:	bit_data = irdata_i[4];
	3'b101:	bit_data = irdata_i[5];
	3'b110:	bit_data = irdata_i[6];
	3'b111:	bit_data = irdata_i[7];
	endcase
end	// always comb

always @(pbus_mux or effaddr) begin

	case(pbus_mux)
	`PMUX_RD_EA: begin
		paddr = effaddr;
		pen = 1'b1;
	end
	default: begin
		paddr = 16'h0000;
		pen = 1'b0;
	end
	endcase
end	// always comb

always @(xbus_mux or dptr or p2 or ri_addr or acc) begin
	xaddr = 16'h0000;
	xen = 1'b0;
	xwdata = 8'h00;
	xwen = 1'b0;

	case(xbus_mux)
	`XMUX_DPTR_RD: begin
		xaddr = dptr;
		xen = 1'b1;
	end
	`XMUX_RI_RD: begin
		xaddr = {p2, ri_addr};
		xen = 1'b1;
	end
	`XMUX_DPTR_WR: begin
		xaddr = dptr;
		xen = 1'b1;
		xwdata = acc;
		xwen = 1'b1;
	end
	`XMUX_RI_WR: begin
		xaddr = {p2, ri_addr};
		xen = 1'b1;
		xwdata = acc;
		xwen = 1'b1;
	end
	endcase
end	// always comb

endmodule


// End of ac51_exe_data.v
