//
// ac51_exe_alu.v
//
// ac51 microcontroller core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac51_defs.v"

module ac51_exe_alu(
	op_sel,
	ina,
	inb,
	div_tmp_i,
	cy_bit,
	ac_bit,
	ov_bit,
	outa,
	outb,
	div_tmp_o,
	new_cy,
	new_ac,
	new_ov
);

input [`ALUOP_LEN - 1:0]	op_sel;
input [7:0]	ina;
input [7:0]	inb;
input [15:0]	div_tmp_i;
input	cy_bit;
input	ac_bit;
input	ov_bit;
output [7:0]	outa;
output [7:0]	outb;
output [15:0]	div_tmp_o;
output	new_cy;
output	new_ac;
output	new_ov;

reg [7:0]	outa;
reg [7:0]	outb;
wire [15:0]	div_tmp_o;
reg	new_cy;
reg	new_ac;
reg	new_ov;

reg [7:0]	bit_msk;
reg	bit_data;

reg [1:0]	addsub_op;
wire [7:0]	addsub_out;
wire	addsub_ncy;
wire	addsub_nac;
wire	addsub_nov;

wire [15:0]	mul_out;

reg [1:0]	div_op;
wire [7:0]	div_outa;
wire [7:0]	div_outb;
wire	div_new_ov;

wire [7:0]	daa_out;
wire	daa_new_cy;

always @(
	op_sel or ina or inb or cy_bit or ac_bit or ov_bit
	or bit_msk or bit_data
	or addsub_out or addsub_ncy or addsub_nac or addsub_nov
	or mul_out or div_outa or div_outb or div_new_ov
	or daa_out or daa_new_cy
) begin
	outa = 8'h00;
	outb = 8'h00;
	new_cy = cy_bit;
	new_ac = ac_bit;
	new_ov = ov_bit;
	addsub_op = 2'b00;
	div_op = 2'b00;

	case(op_sel)
	`ALUOP_ADD: begin
		addsub_op = 2'b00;
		outa = addsub_out;
		new_cy = addsub_ncy;
		new_ac = addsub_nac;
		new_ov = addsub_nov;
	end
	`ALUOP_ADC: begin
		addsub_op = 2'b01;
		outa = addsub_out;
		new_cy = addsub_ncy;
		new_ac = addsub_nac;
		new_ov = addsub_nov;
	end
	`ALUOP_SUB: begin
		addsub_op = 2'b10;
		outa = addsub_out;
		new_cy = addsub_ncy;
		new_ac = addsub_nac;
		new_ov = addsub_nov;
	end
	`ALUOP_SBB: begin
		addsub_op = 2'b11;
		outa = addsub_out;
		new_cy = addsub_ncy;
		new_ac = addsub_nac;
		new_ov = addsub_nov;
	end
	`ALUOP_AND: begin
		outa = ina & inb;
	end
	`ALUOP_BSET: begin
		outa = ina | bit_msk;
		new_cy = 1'b1;	// for SETB C
	end
	`ALUOP_BCLR: begin
		outa = ina & ~bit_msk;
		new_cy = 1'b0;	// for CLR C
	end
	`ALUOP_BCOM: begin
		outa = ina ^ bit_msk;
		new_cy = ~cy_bit;	// for CPL C
	end
	`ALUOP_COM: begin
		outa = ~ina;
	end
	`ALUOP_SWP: begin
		outa = {ina[3:0], ina[7:4]};
	end
	`ALUOP_OR: begin
		outa = ina | inb;
	end
	`ALUOP_XOR: begin
		outa = ina ^ inb;
	end
	`ALUOP_MUL: begin
		outa = mul_out[7:0];
		outb = mul_out[15:8];
		new_cy = 1'b0;
		new_ov = (| mul_out[15:8]);
	end
	`ALUOP_DIV0: begin
		div_op = 2'b01;
		outa = div_outa;
		outb = div_outb;
		new_cy = 1'b0;
		new_ov = div_new_ov;
	end
	`ALUOP_DIV1: begin
		div_op = 2'b10;
		outa = div_outa;
		outb = div_outb;
		new_cy = 1'b0;
		new_ov = div_new_ov;
	end
	`ALUOP_DAA: begin
		outa = daa_out;
		new_cy = daa_new_cy;
	end
	`ALUOP_RR: begin
		outa = {ina[0], ina[7:1]};
	end
	`ALUOP_RRC: begin
		outa = {cy_bit, ina[7:1]};
		new_cy = ina[0];
	end
	`ALUOP_RL: begin
		outa = {ina[6:0], ina[7]};
	end
	`ALUOP_RLC: begin
		outa = {ina[6:0], cy_bit};
		new_cy = ina[7];
	end
	`ALUOP_XCHD: begin
		outa = {ina[7:4], inb[3:0]};
		outb = {inb[7:4], ina[3:0]};
	end
	`ALUOP_BAND: begin
		new_cy = cy_bit & bit_data;
	end
	`ALUOP_BOR: begin
		new_cy = cy_bit | bit_data;
	end
	`ALUOP_BANDNOT: begin
		new_cy = cy_bit & ~bit_data;
	end
	`ALUOP_BORNOT: begin
		new_cy = cy_bit | ~bit_data;
	end
	`ALUOP_BMOV: begin
		new_cy = bit_data;
		if(cy_bit) outa = ina | bit_msk;
		else outa = ina & ~bit_msk;
	end
	endcase
end	// always comb

always @(inb) begin
	bit_msk = 8'h00;

	case(inb[2:0])
	3'b000:	bit_msk = 8'h01;
	3'b001:	bit_msk = 8'h02;
	3'b010:	bit_msk = 8'h04;
	3'b011:	bit_msk = 8'h08;
	3'b100:	bit_msk = 8'h10;
	3'b101:	bit_msk = 8'h20;
	3'b110:	bit_msk = 8'h40;
	3'b111:	bit_msk = 8'h80;
	endcase
end	// always comb

always @(inb or ina) begin
	bit_data = 1'b0;

	case(inb[2:0])
	3'b000:	bit_data = ina[0];
	3'b001:	bit_data = ina[1];
	3'b010:	bit_data = ina[2];
	3'b011:	bit_data = ina[3];
	3'b100:	bit_data = ina[4];
	3'b101:	bit_data = ina[5];
	3'b110:	bit_data = ina[6];
	3'b111:	bit_data = ina[7];
	endcase
end	// always comb

ac51_addsub addsub0(
	.op(addsub_op),
	.ina(ina),
	.inb(inb),
	.cy_bit(cy_bit),
	.out(addsub_out),
	.new_cy(addsub_ncy),
	.new_ac(addsub_nac),
	.new_ov(addsub_nov)
);

ac51_mul mul0(
	.ina(ina),
	.inb(inb),
	.out(mul_out)
);

ac51_div div0(
	.div_op(div_op),
	.ina(ina),
	.inb(inb),
	.tmp_i(div_tmp_i),
	.ov_bit(ov_bit),
	.outa(div_outa),
	.outb(div_outb),
	.tmp_o(div_tmp_o),
	.new_ov(div_new_ov)
);

ac51_daa daa0(
	.in(ina),
	.cy_bit(cy_bit),
	.ac_bit(ac_bit),
	.out(daa_out),
	.new_cy(daa_new_cy)
);

endmodule

// End of ac51_exe_alu.v
