//
// ac09_mcon.v
//
// ac09 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

module ac09_mcon(
	clk,
	rst,
	addr,
	en,
	size,
	wen,
	wdata,
	rdata,
	ack,
	addr8,
	en8,
	wen8,
	wdata8,
	rdata8,
	ack8
);

input	clk;
input	rst;
input [15:0]	addr;
input	en;
input	size;
input	wen;
input [15:0]	wdata;
output [15:0]	rdata;
output	ack;
output [15:0]	addr8;
output	en8;
output	wen8;
output [7:0]	wdata8;
input [7:0]	rdata8;
input	ack8;

reg [15:0]	rdata;
reg [7:0]	wdata8;

reg	bus_act;
wire	smp_tim;
reg [15:0]	addr_r;
reg	en_r;
reg	size_r;
reg	wen_r;
reg [15:0]	wdata_r;
wire [15:0]	addr_s;
wire	en_s;
wire	size_s;
wire	wen_s;
wire [15:0]	wdata_s;

wire	read8_en;
wire	read16_en;
reg	read8_en_d;
reg	read16_en_d;
wire	write8_en;
wire	write16_en;
wire [15:0]	addr_plus_1;
reg	wd2;
reg	wd2_nxt;
reg	last_wd;
reg [7:0]	tmp;
wire	wrt_tmp;


always @(posedge clk or negedge rst) begin
	if(~rst)
		bus_act <= 1'b0;
	else if(en)
		bus_act <= 1'b1;
	else if(ack)
		bus_act <= 1'b0;
end

assign	smp_tim = (~bus_act & en) | (bus_act & ack);

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		addr_r <= 16'h0000;
		en_r <= 1'b0;
		size_r <= 1'b0;	// 8bit
		wen_r <= 1'b0;
		wdata_r <= 16'h0000;
	end
	else if(smp_tim) begin
		addr_r <= addr;
		en_r <= en;
		size_r <= size;
		wen_r <= wen;
		wdata_r <= wdata;
	end
end

// stretched signals
assign	addr_s = smp_tim ? addr : addr_r;
assign	en_s = smp_tim ? en : en_r;
assign	size_s = smp_tim ? size : size_r;
assign	wen_s = smp_tim ? wen : wen_r;
assign	wdata_s = smp_tim ? wdata : wdata_r;

assign	read8_en = (size_s == 1'b0) & en_s & ~wen_s;
assign	read16_en = (size_s == 1'b1) & en_s & ~wen_s;
assign	write8_en = (size_s == 1'b0) & en_s & wen_s;
assign	write16_en = (size_s == 1'b1) & en_s & wen_s;

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		read8_en_d <= 1'b0;
		read16_en_d <= 1'b0;
	end
	else begin
		read8_en_d <= read8_en;
		read16_en_d <= read16_en;
	end
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		tmp <= 8'h00;
	else if(wrt_tmp)
		tmp <= rdata8;
end

assign	wrt_tmp = read16_en & ack8;

always @(posedge clk or negedge rst) begin
	if(~rst)
		wd2 <= 1'b0;
	else
		wd2 <= wd2_nxt;
end

always @(size_s or en_s or smp_tim or ack8 or wd2) begin
	if((size_s == 1'b1) & en_s) begin
		if(smp_tim)
			wd2_nxt = 1'b0;
		else if(ack8)
			wd2_nxt = ~wd2;
		else
			wd2_nxt = wd2;
	end
	else begin
		wd2_nxt = 1'b0;
	end
end	// always comb

assign	addr_plus_1 = addr_s +16'h0001;

assign	addr8 = wd2_nxt ? addr_plus_1 : addr_s;

always @(posedge clk or negedge rst) begin
	if(~rst)
		last_wd <= 1'b1;
	else if(~last_wd & ack8)
		last_wd <= 1'b1;
	else if(en & size == 1'b1)
		last_wd <= 1'b0;
end

assign	en8 = en_s;

always @(write8_en or write16_en or wd2_nxt or wdata_s) begin
	if(write8_en) begin
		wdata8 = wdata_s[7:0];
	end
	else if(write16_en) begin
		if(wd2_nxt) wdata8 = wdata_s[7:0];
		else wdata8 = wdata_s[15:8];
	end
	else begin
		wdata8 = 8'h00;
	end
end	// always comb

always @(
	read8_en_d or read16_en_d or rdata8 or tmp
) begin
	if(read8_en_d)
		rdata = {8'h00, rdata8};
	else if(read16_en_d)
		rdata = {tmp, rdata8};
	else
		rdata = 16'h0000;
end	// always comb

assign	wen8 = write8_en | write16_en;

assign	ack = last_wd & ack8;

endmodule

// End of ac09_mcon.v
