//
// ac09_exe_data.v
//
// ac09 processor core
//
// Version 0.6
//
// Copyright 2008, Hideyuki Abe. All rights reserved.
// Distributed under the terms of the MIT License.
//

`include "ac09_defs.v"

module ac09_exe_data(
	clk,
	rst,
	inst_ack,
	pc,
	new_pc,
	opcode,
	opcode_r,
	main_opr,
	main_opr_r,
	cond,
	cond_ok,
	rlst_end,
	pc_wen,
	opreg,
	amode,
	amode_r,
	postbyte,
	postbyte_r,
	oprnd,
	cfg_mux,
	ea_mux,
	pc_mux,
	cpc_mux,
	tmp_mux,
	ina_mux,
	inb_mux,
	cc_mux,
	h_bit,
	c_bit,
	v_bit,
	new_h,
	new_n,
	new_z,
	new_v,
	new_c,
	opsize,
	ina,
	inb,
	outa,
	outb,
	rbus_mux,
	rnum_mux,
	rwd_mux,
	dbus_mux,
	da_mux,
	dwd_mux,
	daddr,
	den,
	dsize,
	dwen,
	dwdata,
	drdata,
	irq,
	fiq,
	nmi,
	irq_i,
	fiq_i,
	nmi_i,
	sync_wup
);

input	clk;
input	rst;
input	inst_ack;
input [15:0]	pc;
output [15:0]	new_pc;
input [`OPC_LEN - 1:0]	opcode;
output [`OPC_LEN - 1:0]	opcode_r;
input [`ALUOP_LEN - 1:0]	main_opr;
output [`ALUOP_LEN - 1:0]	main_opr_r;
input [3:0]	cond;
output	cond_ok;
output	rlst_end;
output	pc_wen;
input [`REG_LEN - 1:0]	opreg;
input [`AMODE_LEN - 1:0]	amode;
output [`AMODE_LEN - 1:0]	amode_r;
input [7:0]	postbyte;
output [7:0]	postbyte_r;
input [15:0]	oprnd;
input [`CFGMUX_LEN - 1:0]	cfg_mux;
input [`EAMUX_LEN - 1:0]	ea_mux;
input [`PCMUX_LEN - 1:0]	pc_mux;
input [`CPCMUX_LEN - 1:0]	cpc_mux;
input [`TMUX_LEN - 1:0]	tmp_mux;
input [`AMUX_LEN - 1:0]	ina_mux;
input [`BMUX_LEN - 1:0]	inb_mux;
input [`CCMUX_LEN - 1:0]	cc_mux;
output	h_bit;
output	c_bit;
output	v_bit;
input	new_h;
input	new_n;
input	new_z;
input	new_v;
input	new_c;
output	opsize;
output [15:0]	ina;
output [15:0]	inb;
input [15:0]	outa;
input [15:0]	outb;
input [`RMUX_LEN - 1:0]	rbus_mux;
input [`RNUM_LEN - 1:0]	rnum_mux;
input [`RWDMUX_LEN - 1:0]	rwd_mux;
input [`DMUX_LEN - 1:0]	dbus_mux;
input [`DAMUX_LEN - 1:0]	da_mux;
input [`DWDMUX_LEN - 1:0]	dwd_mux;
output [15:0]	daddr;
output	den;
output	dsize;
output	dwen;
output [15:0]	dwdata;
input [15:0]	drdata;
input	irq;
input	fiq;
input	nmi;
output	irq_i;
output	fiq_i;
output	nmi_i;
output	sync_wup;

reg [15:0]	pc_r;
reg [`OPC_LEN - 1:0]	opcode_r;
reg [`ALUOP_LEN - 1:0]	main_opr_r;
reg [3:0]	cond_r;
reg	cond_ok;
reg [`REG_LEN - 1:0]	opreg_r;
reg [`AMODE_LEN - 1:0]	amode_r;
reg [7:0]	postbyte_r;
reg [15:0]	oprnd_r;

reg [15:0]	cur_pc;
reg [15:0]	cur_pc_nxt;

reg [15:0]	effaddr;
reg [15:0]	effaddr_nxt;
reg [15:0]	effaddr_nxt2;

reg [15:0]	ina;
reg [15:0]	inb;

reg	h_bit;
reg	n_bit;
reg	z_bit;
reg	v_bit;
reg	c_bit;
reg	e_bit;
reg	f_bit;
reg	i_bit;
reg	h_nxt;
reg	n_nxt;
reg	z_nxt;
reg	v_nxt;
reg	c_nxt;
reg	e_nxt;
reg	f_nxt;
reg	i_nxt;
wire [7:0]	ccr;
wire [7:0]	ccr_nxt;

wire	irq_i;
wire	fiq_i;
wire	nmi_i;

wire [7:0]	dpr;
wire [15:0]	accd;
wire [15:0]	ibase;

reg [`REG_LEN - 1:0]	raddr;
reg	ren;
reg	rwen;
reg [15:0]	rwdata;
wire [15:0]	rrdata;

reg [15:0]	daddr_i;
reg	den_i;
reg	dsize_i;
reg	dwen_i;
reg [15:0]	dwdata_i;
wire [15:0]	drdata_i;
wire	ack_i;

reg [15:0]	new_pc;
wire [15:0]	pc_o;

reg [15:0]	tmp;
reg [15:0]	tmp_nxt;

wire	cfg_fiq;
wire	cfg_irq;
wire	cfg_rti;


assign	cfg_fiq = (cfg_mux == `CFGMUX_FIQ);
assign	cfg_irq = (cfg_mux == `CFGMUX_IRQ);
assign	cfg_rti = (cfg_mux == `CFGMUX_RTI);

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		pc_r <= 16'h0000;
		opcode_r <= `OPC_NOP;
		main_opr_r <= `ALUOP_NONE;
		cond_r <= 4'b0000;
		opreg_r <= `REG_ACCD;
		amode_r <= `AMODE_NONE;
		postbyte_r <= 8'h00;
		oprnd_r <= 16'h0000;
	end
	else if(inst_ack) begin
		pc_r <= pc;
		opcode_r <= opcode;
		main_opr_r <= main_opr;
		cond_r <= cond;
		opreg_r <= opreg;
		amode_r <= amode;
		postbyte_r <= postbyte;
		oprnd_r <= oprnd;
	end
	else if(cfg_fiq) begin
		opreg_r <= `REG_SSP;
		postbyte_r <= 8'h01;	// CCR
	end
	else if(cfg_irq) begin
		opreg_r <= `REG_SSP;
		postbyte_r <= 8'h7f;	// all regs except PC
	end
	else if(cfg_rti) begin
		if(drdata_i[7]) postbyte_r <= 8'hfe;	// all regs except CCR
		else postbyte_r <= 8'h80;	// PC
	end
end

always @(posedge clk or negedge rst) begin
	if(~rst)
		cur_pc <= 16'h0000;
	else
		cur_pc <= cur_pc_nxt;
end

always @(cur_pc or cpc_mux or pc_r or pc) begin
	case(cpc_mux)
	`CPCMUX_PCR:
		cur_pc_nxt = pc_r;
	`CPCMUX_PCN:
		cur_pc_nxt = pc;
	default:
		cur_pc_nxt = cur_pc;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		effaddr <= 16'h0000;
	else
		effaddr <= effaddr_nxt2;
end

always @(
	effaddr_nxt or inst_ack or amode or dpr or oprnd
) begin
	if(inst_ack) begin
		case(amode)
		`AMODE_DIR:
			effaddr_nxt2 = {dpr, oprnd[7:0]};
		`AMODE_EXT:
			effaddr_nxt2 = oprnd;	// data
		default:
			effaddr_nxt2 = effaddr_nxt;
		endcase
	end
	else begin
		effaddr_nxt2 = effaddr_nxt;
	end
end	// always comb

always @(
	ea_mux or effaddr or outa or drdata_i or ibase
) begin
	case(ea_mux)
	`EAMUX_OUTA:
		effaddr_nxt = outa;
	`EAMUX_DDAT:
		effaddr_nxt = drdata_i;
	`EAMUX_IBASE:
		effaddr_nxt = ibase;
	default:
		effaddr_nxt = effaddr;
	endcase
end	// always comb

always @(
	pc_mux or tmp or outa or effaddr or pc_o
) begin
	case(pc_mux)
	`PCMUX_TMP:
		new_pc = tmp;
	`PCMUX_OUTA:
		new_pc = outa;
	`PCMUX_EA:
		new_pc = effaddr[15:0];
	`PCMUX_PCO:
		new_pc = pc_o;
	default:	// PCMUX_ZERO
		new_pc = 16'h0000;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst)
		tmp <= 16'h0000;
	else
		tmp <= tmp_nxt;
end

always @(
	tmp_mux or tmp or drdata_i or rrdata
	or postbyte_r
) begin
	case(tmp_mux)
	`TMUX_DDAT:
		tmp_nxt = drdata_i;
	`TMUX_RDAT:
		tmp_nxt = rrdata;
	`TMUX_PSHINI:
		if(postbyte_r[7]) tmp_nxt = 16'h0007;
		else if(postbyte_r[6]) tmp_nxt = 16'h0006;
		else if(postbyte_r[5]) tmp_nxt = 16'h0005;
		else if(postbyte_r[4]) tmp_nxt = 16'h0004;
		else if(postbyte_r[3]) tmp_nxt = 16'h0003;
		else if(postbyte_r[2]) tmp_nxt = 16'h0002;
		else if(postbyte_r[1]) tmp_nxt = 16'h0001;
		else if(postbyte_r[0]) tmp_nxt = 16'h0000;
		else tmp_nxt = 16'h00f0;
	`TMUX_PSHNXT: begin : psh_rlst
		tmp_nxt = tmp;
		if(tmp_nxt[2:0] == 3'd7) begin
			tmp_nxt = 16'h0006;
			if(postbyte_r[6]) disable psh_rlst;
		end
		if(tmp_nxt[2:0] == 3'd6) begin
			tmp_nxt = 16'h0005;
			if(postbyte_r[5]) disable psh_rlst;
		end
		if(tmp_nxt[2:0] == 3'd5) begin
			tmp_nxt = 16'h0004;
			if(postbyte_r[4]) disable psh_rlst;
		end
		if(tmp_nxt[2:0] == 3'd4) begin
			tmp_nxt = 16'h0003;
			if(postbyte_r[3]) disable psh_rlst;
		end
		if(tmp_nxt[2:0] == 3'd3) begin
			tmp_nxt = 16'h0002;
			if(postbyte_r[2]) disable psh_rlst;
		end
		if(tmp_nxt[2:0] == 3'd2) begin
			tmp_nxt = 16'h0001;
			if(postbyte_r[1]) disable psh_rlst;
		end
		if(tmp_nxt[2:0] == 3'd1) begin
			tmp_nxt = 16'h0000;
			if(postbyte_r[0]) disable psh_rlst;
		end
		tmp_nxt = 16'h00f0;
	end	// psh_rlst
	`TMUX_PULINI:
		if(postbyte_r[0]) tmp_nxt = 16'h0000;
		else if(postbyte_r[1]) tmp_nxt = 16'h0001;
		else if(postbyte_r[2]) tmp_nxt = 16'h0002;
		else if(postbyte_r[3]) tmp_nxt = 16'h0003;
		else if(postbyte_r[4]) tmp_nxt = 16'h0004;
		else if(postbyte_r[5]) tmp_nxt = 16'h0005;
		else if(postbyte_r[6]) tmp_nxt = 16'h0006;
		else if(postbyte_r[7]) tmp_nxt = 16'h0007;
		else tmp_nxt = 16'h00f0;
	`TMUX_PULNXT: begin : pul_rlst
		tmp_nxt = tmp;
		if(tmp_nxt[2:0] == 3'd0) begin
			tmp_nxt = 16'h0001;
			if(postbyte_r[1]) disable pul_rlst;
		end
		if(tmp_nxt[2:0] == 3'd1) begin
			tmp_nxt = 16'h0002;
			if(postbyte_r[2]) disable pul_rlst;
		end
		if(tmp_nxt[2:0] == 3'd2) begin
			tmp_nxt = 16'h0003;
			if(postbyte_r[3]) disable pul_rlst;
		end
		if(tmp_nxt[2:0] == 3'd3) begin
			tmp_nxt = 16'h0004;
			if(postbyte_r[4]) disable pul_rlst;
		end
		if(tmp_nxt[2:0] == 3'd4) begin
			tmp_nxt = 16'h0005;
			if(postbyte_r[5]) disable pul_rlst;
		end
		if(tmp_nxt[2:0] == 3'd5) begin
			tmp_nxt = 16'h0006;
			if(postbyte_r[6]) disable pul_rlst;
		end
		if(tmp_nxt[2:0] == 3'd6) begin
			tmp_nxt = 16'h0007;
			if(postbyte_r[7]) disable pul_rlst;
		end
		tmp_nxt = 16'h00f0;
	end	// pul_rlst
	default:	// TMUX_NULL
		tmp_nxt = tmp;
	endcase
end	// always comb

assign	rlst_end = |tmp[7:4];

assign	opsize = ~(opreg_r[3]);

always @(
	ina_mux or oprnd_r or rrdata or pc_r or drdata_i
	or effaddr or ibase
) begin
	case(ina_mux)
	`AMUX_OPRND:
		ina = oprnd_r;
	`AMUX_RDAT:
		ina = rrdata;
	`AMUX_PC:
		ina = pc_r;
	`AMUX_DDAT:
		ina = drdata_i;
	`AMUX_EA:
		ina = effaddr;
	`AMUX_IBASE:
		ina = ibase;
	default:
		ina = 16'h0000;
	endcase
end	// always comb

always @(
	inb_mux or oprnd_r or drdata_i or postbyte_r or accd
	or tmp
) begin
	case(inb_mux)
	`BMUX_OPRND:
		inb = oprnd_r;
	`BMUX_OPRND8SE:
		inb = {{8{oprnd_r[7]}},oprnd_r[7:0]};
	`BMUX_DDAT:
		inb = drdata_i;
	`BMUX_OFF5:
		inb = {{11{postbyte_r[4]}},postbyte_r[4:0]};
	`BMUX_ACCA:
		inb = {{8{accd[15]}},accd[15:8]};
	`BMUX_ACCB:
		inb = {{8{accd[7]}},accd[7:0]};
	`BMUX_ACCD:
		inb = accd;
	`BMUX_ONE:
		inb = 16'h0001;
	`BMUX_TWO:
		inb = 16'h0002;
	`BMUX_ACCBZE:
		inb = {8'h00, accd[7:0]};
	`BMUX_TMP:
		inb = tmp;
	default:
		inb = 16'h0000;
	endcase
end	// always comb

always @(posedge clk or negedge rst) begin
	if(~rst) begin
		h_bit <= 1'b0;
		n_bit <= 1'b0;
		z_bit <= 1'b0;
		v_bit <= 1'b0;
		c_bit <= 1'b0;
		e_bit <= 1'b0;
		f_bit <= 1'b1;
		i_bit <= 1'b1;
	end
	else begin
		h_bit <= h_nxt;
		n_bit <= n_nxt;
		z_bit <= z_nxt;
		v_bit <= v_nxt;
		c_bit <= c_nxt;
		e_bit <= e_nxt;
		f_bit <= f_nxt;
		i_bit <= i_nxt;
	end
end

assign	ccr = {e_bit,f_bit,h_bit,i_bit,n_bit,z_bit,v_bit,c_bit};

always @(
	cc_mux or ccr_nxt
	or new_h or new_n or new_z or new_v or new_c
) begin
	h_nxt = ccr_nxt[5];
	n_nxt = ccr_nxt[3];
	z_nxt = ccr_nxt[2];
	v_nxt = ccr_nxt[1];
	c_nxt = ccr_nxt[0];
	e_nxt = ccr_nxt[7];
	f_nxt = ccr_nxt[6];
	i_nxt = ccr_nxt[4];

	case(cc_mux)
	`CCMUX_LOP: begin
		n_nxt = new_n;
		z_nxt = new_z;
		v_nxt = new_v;
	end
	`CCMUX_AOP: begin
		h_nxt = new_h;
		n_nxt = new_n;
		z_nxt = new_z;
		v_nxt = new_v;
		c_nxt = new_c;
	end
	`CCMUX_AOP2: begin
		n_nxt = new_n;
		z_nxt = new_z;
		v_nxt = new_v;
		c_nxt = new_c;
	end
	`CCMUX_ZBIT: begin
		z_nxt = new_z;
	end
	`CCMUX_AOP3: begin
		z_nxt = new_z;
		c_nxt = new_c;
	end
	`CCMUX_MSKF: begin
		f_nxt = 1'b1;
		i_nxt = 1'b1;
	end
	`CCMUX_MSKI: begin
		i_nxt = 1'b1;
	end
	`CCMUX_SETE: begin
		e_nxt = 1'b1;
	end
	`CCMUX_CLRE: begin
		e_nxt = 1'b0;
	end
	endcase
end

always @(
	cond_r or n_bit or z_bit or v_bit or c_bit
) begin
	case(cond_r[3:1])
	3'b001:		// bls (c|z = 1)
		cond_ok = (c_bit | z_bit);
	3'b010:		// bcs
		cond_ok = c_bit;
	3'b011:		// beq
		cond_ok = z_bit;
	3'b100:		// bvs
		cond_ok = v_bit;
	3'b101:		// bmi
		cond_ok = n_bit;
	3'b110:		// blt (n^v = 1)
		cond_ok = (n_bit ^ v_bit);
	3'b111:		// ble (z|(n^v) = 1)
		cond_ok = (z_bit | (n_bit ^ v_bit));
	default:	// never
		cond_ok = 1'b0;
	endcase

	if(cond_r[0] == 1'b0)
		cond_ok = ~cond_ok;
end	// always comb

assign	irq_i = i_bit ? 1'b0 : irq;
assign	fiq_i = f_bit ? 1'b0 : fiq;
assign	nmi_i = nmi;
assign	sync_wup = irq | fiq | nmi;

always @(rbus_mux) begin
	ren = 1'b0;
	rwen = 1'b0;

	case(rbus_mux)
	`RMUX_WR: begin
		ren = 1'b1;
		rwen = 1'b1;
	end
	`RMUX_RD: begin
		ren = 1'b1;
	end
	endcase
end	// always comb

always @(
	rnum_mux or opreg_r or oprnd_r or postbyte_r
	or tmp
) begin
	raddr = 4'h0;

	case(rnum_mux)
	`RNUM_OPREG:
		raddr = opreg_r;
	`RNUM_REG1:
		raddr = oprnd_r[7:4];
	`RNUM_REG2:
		raddr = oprnd_r[3:0];
	`RNUM_IBASE:
		case(postbyte_r[6:5])
		2'b00:
			raddr = `REG_IX;
		2'b01:
			raddr = `REG_IY;
		2'b10:
			raddr = `REG_USP;
		default:	// 2'b11
			raddr = `REG_SSP;
		endcase
	`RNUM_RLST:
		case(tmp[2:0])
		3'b000:		raddr = `REG_CCR;
		3'b001:		raddr = `REG_ACCA;
		3'b010:		raddr = `REG_ACCB;
		3'b011:		raddr = `REG_DPR;
		3'b100:		raddr = `REG_IX;
		3'b101:		raddr = `REG_IY;
		3'b110:		if(opreg_r == `REG_USP) raddr = `REG_SSP;
					else raddr = `REG_USP;
		default:	raddr = `REG_PC;
		endcase
	endcase
end	// always comb

always @(
	rwd_mux or outa or rrdata or tmp
	or effaddr or drdata_i
) begin
	case(rwd_mux)
	`RWDMUX_OUTA:
		rwdata = outa;
	`RWDMUX_RDAT:
		rwdata = rrdata;
	`RWDMUX_TMP:
		rwdata = tmp;
	`RWDMUX_EA:
		rwdata = effaddr;
	`RWDMUX_DDAT:
		rwdata = drdata_i;
	default:
		rwdata = 16'h0000;
	endcase
end	// always comb

ac09_exe_regfile exe_regf0(
	.clk(clk),
	.rst(rst),
	.raddr(raddr),
	.ren(ren),
	.rwen(rwen),
	.rwdata(rwdata),
	.rrdata(rrdata),
	.dpr_o(dpr),
	.accd_o(accd),
	.ccr_i(ccr),
	.ccr_o(ccr_nxt),
	.pc_i(pc_r),
	.pc_o(pc_o),
	.pc_wen(pc_wen),
	.ibase_sel(postbyte_r[6:5]),
	.ibase(ibase)
);

always @(
	dbus_mux or opreg_r or tmp
) begin
	den_i = 1'b0;
	dsize_i = 1'b0;	// 8bit
	dwen_i = 1'b0;

	case(dbus_mux)
	`DMUX_RD16: begin
		den_i = 1'b1;
		dsize_i = 1'b1;
	end
	`DMUX_RD: begin
		den_i = 1'b1;
		if(opreg_r[3] == 1'b0)	// 16bit
			dsize_i = 1'b1;
	end
	`DMUX_WR: begin
		den_i = 1'b1;
		if(opreg_r[3] == 1'b0)	// 16bit
			dsize_i = 1'b1;
		dwen_i = 1'b1;
	end
	`DMUX_WR16: begin
		den_i = 1'b1;
		dsize_i = 1'b1;
		dwen_i = 1'b1;
	end
	`DMUX_WRT: begin
		den_i = 1'b1;
		dsize_i = tmp[2];
		dwen_i = 1'b1;
	end
	`DMUX_RDT: begin
		den_i = 1'b1;
		dsize_i = tmp[2];
	end
	endcase
end	// always comb

always @(
	da_mux or effaddr or outa or rrdata
) begin
	case(da_mux)
	`DAMUX_RSTV:
		daddr_i = 16'hfffe;
	`DAMUX_EA:
		daddr_i = effaddr;
	`DAMUX_OUTA:
		daddr_i = outa;
	`DAMUX_RDAT:
		daddr_i = rrdata;
	`DAMUX_NMIV:
		daddr_i = 16'hfffc;
	`DAMUX_FIQV:
		daddr_i = 16'hfff6;
	`DAMUX_IRQV:
		daddr_i = 16'hfff8;
	`DAMUX_SWIV:
		daddr_i = 16'hfffa;
	`DAMUX_SWI2V:
		daddr_i = 16'hfff4;
	`DAMUX_SWI3V:
		daddr_i = 16'hfff2;
	default:	// DAMUX_ZERO
		daddr_i = 16'h0000;
	endcase
end	// always comb

always @(
	dwd_mux or outa or pc_r or rrdata
	or cur_pc
) begin
	case(dwd_mux)
	`DWDMUX_OUTA:
		dwdata_i = outa;
	`DWDMUX_PC:
		dwdata_i = pc_r;
	`DWDMUX_RDAT:
		dwdata_i = rrdata;
	`DWDMUX_CPC:
		dwdata_i = cur_pc;
	default:
		dwdata_i = 16'h0000;
	endcase
end	// always comb

assign	daddr = daddr_i;
assign	den = den_i;
assign	dsize = dsize_i;
assign	dwen = dwen_i;
assign	dwdata = dwdata_i;
assign	drdata_i = drdata;

endmodule

// End of ac09_exe_data.v
